// mlsegclock.cpp
//
//   A segment clock widget.
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//    $Id: mlsegclock.cpp,v 1.2 2007/09/04 11:35:14 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU Library General Public License 
//   version 2 as published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qpainter.h>
#include <qpixmap.h>
#include <qdatetime.h>
#include <qfontmetrics.h>
#include <stdio.h>
#include <string.h>

#include <mlsegclock.h>

MLSegClock::MLSegClock(Qt::Orientation style,QWidget *parent,
		     const char *name)  :QWidget(parent,name)
{
  seg_style=style;

  //
  // Create Fonts
  //
  title_font=QFont(MLSEGCLOCK_TITLE_FONT_FACE,
		   MLSEGCLOCK_TITLE_FONT_SIZE,
		   MLSEGCLOCK_TITLE_FONT_WEIGHT);
  title_font.setPixelSize(MLSEGCLOCK_TITLE_FONT_SIZE);

  digit_font=QFont(MLSEGCLOCK_DIGIT_FONT_FACE,
		   MLSEGCLOCK_DIGIT_FONT_SIZE,
		   MLSEGCLOCK_DIGIT_FONT_WEIGHT);
  digit_font.setPixelSize(MLSEGCLOCK_DIGIT_FONT_SIZE);
  QFontMetrics *fm=new QFontMetrics(digit_font);
  clock_size_x=fm->boundingRect("00:00:00").width();
  clock_size_y=fm->boundingRect("00:00:00").height();
  segment_size_x=fm->boundingRect("00:00").width();
  segment_size_y=fm->boundingRect("00:00").height();  
  delete fm;
  seg_show_hour=true;

  label_font=QFont(MLSEGCLOCK_LABEL_FONT_FACE,
		   MLSEGCLOCK_LABEL_FONT_SIZE,
		   MLSEGCLOCK_LABEL_FONT_WEIGHT);
  label_font.setPixelSize(MLSEGCLOCK_LABEL_FONT_SIZE);

  //
  // Setup Style Layout
  //
  switch(style) {
  case Qt::Vertical:
    clock_x=0;
    clock_y=0;
    segment_x=0;
    segment_y=76;
    label_x=4;
    label_y=166;
    break;
  case Qt::Horizontal:
    clock_x=0;
    clock_y=0;
    segment_x=176;
    segment_y=0;
    label_x=4;
    label_y=89;
    break;
 default:
    clock_x=0;
    clock_y=0;
    segment_x=0;
    segment_y=76;
    label_x=4;
    label_y=166;
    break;
  }

  //
  // Clock Stuff
  //
  time_offset=0;
  previous_time=QTime::currentTime().addSecs(time_offset);
  QTimer *timer=new QTimer(this,"timer");
  connect(timer,SIGNAL(timeout()),this,SLOT(tickClock()));
  timer->start(100);
  rseglist=0;
  current_index=-1;
  time_style=1;
  time_offset=0;

  setPalette(QPalette(QColor(0,0,0)));
}


QSizePolicy MLSegClock::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding);
}


QSize MLSegClock::sizeHint() const
{
  switch(seg_style) {
      case Qt::Horizontal:
	return QSize(344,85);
	break;

      case Qt::Vertical:
	return QSize(172,170);
	break;
  }
  return QSize(172,170);
}


bool MLSegClock::showHour() const
{
  return seg_show_hour;
}


void MLSegClock::setShowHour(bool state)
{
  seg_show_hour=state;
}


void MLSegClock::loadSegList(MLSegList *list)
{
  rseglist=list;
}


void MLSegClock::tickClock()
{
  char accum[9];
  char alabel[255];
  QString str;

  current_time=QTime::currentTime().addSecs(time_offset);
  if(current_time.second()!=previous_time.second()) {
    previous_time=current_time;
    //
    // Clock Display
    //
    if(time_style==1) {
      if(seg_show_hour) {
	str=current_time.toString("h:mm:ss");
      }
      else {
	str=current_time.toString("--:mm:ss");
      }
    }
    else {
      if(seg_show_hour) {
	str=current_time.toString("h:mm:ss ap");
      }
      else {
	str=current_time.toString("--:mm:ss ap");
      }
      str=str.left(str.length()-3);
    }
    QPainter p(this);
    p.fillRect(clock_x+1,clock_y+31,169,38,black);
    p.setPen(red);
    p.setBrush(black);
    p.setFont(digit_font);
#ifdef WIN32
    switch(seg_style) {
	case Qt::Vertical:	
	  p.drawText((sizeHint().width()-p.fontMetrics().width(str))/2,
		     clock_y+24+clock_size_y+(38-clock_size_y)/2,str,8);
	  break;
	case Qt::Horizontal:
	  p.drawText((sizeHint().width()/2-p.fontMetrics().width(str))/2,
		     clock_y+24+clock_size_y+(38-clock_size_y)/2,str,8);
	  break;
    }
#else
    switch(seg_style) {
	case Qt::Vertical:
	  p.drawText((sizeHint().width()-p.fontMetrics().width(str))/2,
		     clock_y+32+clock_size_y+(38-clock_size_y)/2,str,8);
	  break;
	case Qt::Horizontal:
	  p.drawText((sizeHint().width()/2-p.fontMetrics().width(str))/2,
		     clock_y+32+clock_size_y+(38-clock_size_y)/2,str,8);
	  break;
    }
#endif  // WIN32
    //
    // Segment and Label Display
    //
    if(rseglist!=0) {
      int index=rseglist->getIndex(current_time.addSecs(-time_offset));
      if(index!=current_index) {
	if(current_index!=-1) {
	  emit(segFinished(current_index));
	}
	if(index!=-1) {
	  emit(segStarted(index));
	}
	current_index=index;
      }
      if(index>=0) {
	sprintf(accum,"%02d:%02d",
		rseglist->getRemaining(index).minute(),
		rseglist->getRemaining(index).second());
		rseglist->getLabel(index,alabel,255);
      }
      else {
	sprintf(accum,"00:00");
	alabel[0]=0;
      }
    }
    else {
      sprintf(accum,"00:00");
      alabel[0]=0;
    }
    p.fillRect(segment_x+1,segment_y+31,169,38,black);
#ifdef WIN32
    p.drawText(segment_x+(169-segment_size_x)/2,
	       segment_y+24+segment_size_y+(38-segment_size_y)/2,accum,5);
#else
    p.drawText(segment_x+(169-segment_size_x)/2,segment_y+32+segment_size_y+
	       (38-segment_size_y)/2,accum,5);
#endif  // WIN32
    p.fillRect(label_x,label_y-18,200,89,black);
    p.setPen(green);
    p.setFont(label_font);
    p.drawText(label_x,label_y,alabel,strlen(alabel));
  }
}


void MLSegClock::set12Hour()
{
  time_style=0;
}


void MLSegClock::set24Hour()
{
  time_style=1;
}


void MLSegClock::setOffset(int secs)
{
  time_offset=secs;
}


void MLSegClock::paintEvent(QPaintEvent *e)
{
  QPainter p(this);
  p.setPen(red);
  p.setBrush(red);

  //
  // Clock Counter
  //
  p.moveTo(clock_x,clock_y);
  p.lineTo(clock_x+170,clock_y);
  p.lineTo(clock_x+170,clock_y+70);
  p.lineTo(clock_x,clock_y+70);
  p.lineTo(clock_x,clock_y);
  p.moveTo(clock_x,clock_y+30);
  p.lineTo(clock_x+170,clock_y+30);
  p.moveTo(clock_x,clock_y+12);
  p.lineTo(clock_x+50,clock_y+12);
  p.moveTo(clock_x+170-50,clock_y+12);
  p.lineTo(clock_x+170,clock_y+12);
  p.moveTo(clock_x,clock_y+18);
  p.lineTo(clock_x+50,clock_y+18);
  p.moveTo(clock_x+170-50,clock_y+18);
  p.lineTo(clock_x+170,clock_y+18);
  p.setFont(title_font);
  p.drawText(clock_x+55,clock_y+25,tr("TIME"),4);

  //
  // Segment Counter
  //
  p.moveTo(segment_x,segment_y);
  p.lineTo(segment_x+170,segment_y);
  p.lineTo(segment_x+170,segment_y+70);
  p.lineTo(segment_x,segment_y+70);
  p.lineTo(segment_x,segment_y);
  p.moveTo(segment_x,segment_y+30);
  p.lineTo(segment_x+170,segment_y+30);
  p.moveTo(segment_x,segment_y+12);
  p.lineTo(segment_x+20,segment_y+12);
  p.moveTo(segment_x+170-20,segment_y+12);
  p.lineTo(segment_x+170,segment_y+12);
  p.moveTo(segment_x,segment_y+18);
  p.lineTo(segment_x+20,segment_y+18);
  p.moveTo(segment_x+170-20,segment_y+18);
  p.lineTo(segment_x+170,segment_y+18);
  p.setFont(title_font);
  p.drawText(segment_x+25,segment_y+25,tr("SEGMENT"),7);
}
