// mlseglist.cpp
//
//   A segment list container class for RSegClock.
//
//   (C) Copyright 2002-2007 Fred Gleason <fredg@paravelsystems.com>
//
//    $Id: mlseglist.cpp,v 1.2 2007/09/04 11:35:14 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU Library General Public License 
//   version 2 as published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qdatetime.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <mlconf.h>

#include <mlseglist.h>


MLSegList::MLSegList()
{
  seglength=0;
  seglist=0;
  seg_offset=0;
}


int MLSegList::offset() const
{
  return seg_offset;
}


void MLSegList::setOffset(int o)
{
  seg_offset=o;
}


int MLSegList::loadSegFile(QString &filename)
{
  int i=1,j=0;
  char accum[255];
  char segment[255];
  int start_hour,start_min,start_sec;
  int finish_hour,finish_min,finish_sec;
  
  sprintf(accum,"Segment%d",i);
  while(GetPrivateProfileString(filename,"Segments",accum,segment,"",254)>=0) {
    if(sscanf(segment,"%d:%d:%d,%d:%d:%d",
	      &start_hour,&start_min,&start_sec,
	      &finish_hour,&finish_min,&finish_sec)!=6) {
      return -1;
    }
    if((start_hour>23)||(start_hour<0)||
       (start_min>59)||(start_min<0)||
       (start_sec>59)||(start_sec<0)||
       (finish_hour>23)||(finish_hour<0)||
       (finish_min>59)||(finish_min<0)||
       (finish_sec>59)||(finish_sec<0)) {
      return -1;
    }  
    j=0;
    while((segment[j]!=',')&&(segment[j++]!=0));
    if(segment[j]==0) {
      return -1;
    }
    j++;
    while((segment[j]!=',')&&(segment[j++]!=0));
    j++;
    addEntry(QTime(start_hour,start_min,start_sec),
	     QTime(finish_hour,finish_min,finish_sec),
	     segment+j);
    sprintf(accum,"Segment%d",++i);

  }
  return 0;
}

MLSegList::~MLSegList()
{
  free(seglist);
}

MLSegList::MLSegList(const MLSegList &old)
{
  seglist=(struct seglist *)malloc(old.seglength*sizeof(struct seglist));
  for(int i=0;i<old.seglength;i++) {
    *(seglist+i)=*(old.seglist+i);
  }
  seglength=old.seglength;
}

int MLSegList::addEntry(QTime start,QTime finish,const char *label)
{
  seglist=(struct seglist *)realloc(seglist,
				    (seglength+1)*sizeof(struct seglist));
  (seglist+seglength)->start=start;
  (seglist+seglength)->finish=finish;
  strcpy((seglist+seglength)->label,label);
  return seglength++;
}


int MLSegList::delEntry(int index)
{
  if(index>=seglength) {    /* Invalid index */
    return -1;
  }
  for(int i=index;i<seglength-1;i++) {
    *(seglist+index)=*(seglist+index+1);
  }
  seglist=(struct seglist *)realloc(seglist,
				    (--seglength)*sizeof(struct seglist));
  
  return 0;
}


int MLSegList::getIndex(QTime time) const
{
  for(int i=0;i<seglength;i++) {
    if((time.addSecs(seg_offset)>=(seglist+i)->start)&&
       (time.addSecs(seg_offset)<(seglist+i)->finish)) {
      return i;
    }
  }
  return -1;
}


QTime MLSegList::getRemaining(int index) const
{
  int finish_secs=3600*(seglist+index)->finish.hour()+
    60*(seglist+index)->finish.minute()+
    (seglist+index)->finish.second();
  QTime current_time=QTime::currentTime();
  int current_secs=3600*current_time.hour()+
    60*current_time.minute()+
    current_time.second();
  int remaining_secs=finish_secs-current_secs;
  if(remaining_secs<0) {
    remaining_secs+=86400;
  }
  int remaining_hours=remaining_secs/3600;
  int remaining_mins=(remaining_secs-remaining_hours*3600)/60;
  remaining_secs-=remaining_hours*3600+remaining_mins*60;
  return QTime(remaining_hours,remaining_mins,remaining_secs).
    addSecs(-seg_offset);
}


QTime MLSegList::getStart(int index) const
{
  return (seglist+index)->start;
}


QTime MLSegList::getFinish(int index) const
{
  return (seglist+index)->finish;
}


int MLSegList::getLabel(int index,char *label,int length) const
{
  if(strlen((seglist+index)->label)>(unsigned)(length-1)) {
    return -1;
  }
  strcpy(label,(seglist+index)->label);
  return 0;
}


int MLSegList::getSize() const
{
  return seglength;
}
