// mlseglist.cpp
//
//   A segment list container class for MLSegClock.
//
//   (C) Copyright 2002-2007 Fred Gleason <fredg@paravelsystems.com>
//
//    $Id: mlseglist.h,v 1.2 2007/09/04 11:35:14 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU Library General Public License 
//   version 2 as published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifndef MLSEGLIST_H
#define MLSEGLIST_H

#include <qdatetime.h>

struct seglist {
  QTime start;
  QTime finish;
  char label[255];
};

/**
 * @short Broadcast segment list.
 * @author Fred Gleason <fredg@wava.com>
 *
 * This class contains a broadcast segment list, a set of elements each of
 * which consists of a Start Time, a Finish Time and a Label.  A MLSegList
 * is used to provide segment information to a RSegClock widget.  Individual
 * list elements can be accessed by means of index numbers.
 * @see RSegClock
 *
 **/
class MLSegList
{
  public:
  /**
   * Create an empty list.
   **/
    MLSegList();
    ~MLSegList();
    MLSegList(const MLSegList &);
  /**
   * Returns the current time offset in seconds.
   **/
    int offset() const;
  /**
   * Specify the time offset to apply.
   * Where:
   * <pre>
   *  <offset> = Offset in seconds.
   * </pre>
   **/
    void setOffset(int offset);
  /**
   * Loads a .seg file into the list.  Such a file has the following format:
   * <pre>
   * [Segments]
   * SegmentN=<start-time>,<finish-time>,<label>
   * ...
   * </pre>
   * Where:
   * <pre>
   *  <start-time> = Segment start time, in the format HH:MM:SS.  HH should
   *                 be in 24 hour military format. 
   * <finish-time> = Segment finish time, in the format HH:MM:SS.  HH should
   *                 be in 24 hour military format. 
   *       <label> = Segment label to be associated with the time interval.
   * </pre>
   * @returns 0 if successful, -1 if file error encountered. 
   **/
    int loadSegFile(QString &filename);
  /**
   * Add a single entry into the list.
   * @returns The index number of the entry.
   **/
    int addEntry(QTime start,QTime finish,const char *label);
  /**
   * Deletes the referenced entry from the list.
   * @returns 0 if successful, -1 if no such index.
   **/
    int delEntry(int index);
  /**
   * Gets the index of the entry whose range includes the specified time.
   * @returns The corresponding index if it exists, or -1 if none found.
   **/
    int getIndex(QTime time) const;
  /**
   * Get the remaining time until the finish of the specified entry.
   * @returns Time remaining until finish.
   **/
    QTime getRemaining(int index) const;
  /**
   * Get the start time of the referenced entry.
   * @returns The start time, or -1 if entry not found.
   **/
    QTime getStart(int Index) const;
  /**
   * Get the finish time of the referenced entry.
   * @returns The finish time, or -1 if entry not found.
   **/
    QTime getFinish(int index) const;
  /**
   * Get the label of the referenced entry.
   * @returns 0 if successful, or -1 if entry not found.
   **/
    int getLabel(int Index,char *label,int length) const;
  /**
   * Get the total number of entries in the list.
   * @returns Number of entries.
   **/
    int getSize() const;
  
  private:
    struct seglist *seglist;
    int seglength;
    int seg_offset;
};

#endif  // MLSEGLIST_H
