//   mlttydevice_win32.cpp
//
//   The Win32 version of a Qt driver for serial ports.
//
//   (C) Copyright 2002-2007 Fred Gleason <fredg@paravelsystems.com>
//
//    $Id: mlttydevice_win32.cpp,v 1.2 2007/09/04 11:35:14 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU Library General Public License 
//   version 2 as published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//

#include <stdio.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <qiodevice.h>
#include <mlttydevice.h>



MLTTYDevice::MLTTYDevice() : QIODevice()
{
  Init();
}

MLTTYDevice::~MLTTYDevice()
{
  if(tty_open) {
    close();
  }
}

bool MLTTYDevice::open(int mode)
{
  DWORD flags=0;
  WCHAR name[255];
  DCB dcb;
  COMMTIMEOUTS timeouts;

  tty_mode=mode;
  if((mode&IO_ReadWrite)==IO_ReadWrite) {
    flags|=GENERIC_WRITE|GENERIC_READ;
  }
  else {
    if(((mode&IO_WriteOnly)!=0)) {
      flags|=GENERIC_WRITE;
    }
    if(((mode&IO_ReadOnly)!=0)) {
      flags|=GENERIC_READ;
    }
  }
  wcscpy(name,(TCHAR*)qt_winTchar(tty_name,true));
  tty_fd=CreateFile(name,flags,0,NULL,OPEN_EXISTING,
		    FILE_ATTRIBUTE_NORMAL,NULL);
  if(tty_fd==INVALID_HANDLE_VALUE) {
    tty_status=IO_OpenError;
    return false;
  }
  tty_open=true;
  tty_status=IO_Ok;

  SetupComm(tty_fd,WIN32_BUFFER_SIZE,WIN32_BUFFER_SIZE);
  switch(tty_parity) {
  case MLTTYDevice::None:
    BuildCommDCB((TCHAR*)qt_winTchar(QString().
				    sprintf("%d,N,%d,1",
				    tty_speed,tty_length),
				    true),&dcb);
    break;

  case MLTTYDevice::Even:
    BuildCommDCB((TCHAR*)qt_winTchar(QString().
				    sprintf("%d,E,%d,1",
				    tty_speed,tty_length),
				    true),&dcb);
    break;

  case MLTTYDevice::Odd:
    BuildCommDCB((TCHAR*)qt_winTchar(QString().
				    sprintf("%d,O,%d,1",
				    tty_speed,tty_length),
				    true),&dcb);
    break;
  }
  SetCommState(tty_fd,&dcb);
  timeouts.ReadIntervalTimeout=MAXDWORD;
  timeouts.ReadTotalTimeoutMultiplier=0;
  timeouts.ReadTotalTimeoutConstant=0;
  timeouts.WriteTotalTimeoutMultiplier=0;
  timeouts.WriteTotalTimeoutConstant=0;
  SetCommTimeouts(tty_fd,&timeouts);

  return true;
}


void MLTTYDevice::close()
{
  if(tty_open) {
    CloseHandle(tty_fd);
  }
  tty_open=false;
}


void MLTTYDevice::flush()
{
}


Q_LONG MLTTYDevice::readBlock(char *data,Q_ULONG maxlen)
{
  Q_ULONG n;

  if(!ReadFile(tty_fd,data,maxlen,&n,NULL)) {
    if(GetLastError()!=ERROR_TIMEOUT) {
      tty_status=IO_ReadError;
      return -1;
    }
  }
  tty_status=IO_Ok;
  return n;
}


Q_LONG MLTTYDevice::writeBlock(const char *data,Q_ULONG len)
{
  Q_ULONG n;

  if(!WriteFile(tty_fd,data,len,&n,NULL)) {
    tty_status=IO_WriteError;
    return n;
  }
  tty_status=IO_Ok;
  return n;
}


int MLTTYDevice::getch()
{
  char c;
  int n;

  if((n=readBlock(&c,1))<0) {
    tty_status=IO_ReadError;
    return -1;
  }
  return (int)c;
}


int MLTTYDevice::putch(int ch)
{
  char c;
  int n;

  c=(char)ch;
  if((n=writeBlock(&c,1))<0) {
    tty_status=IO_WriteError;
    return -1;
  }
  return ch;
}


int MLTTYDevice::ungetch(int ch)
{
  tty_status=IO_WriteError;
  return -1;
}


QIODevice::Offset MLTTYDevice::size() const
{
  return 0;
}


int MLTTYDevice::flags() const
{
  return tty_mode|state();
}


int MLTTYDevice::mode() const
{
  return tty_mode;
}


int MLTTYDevice::state() const
{
  if(tty_open) {
    return IO_Open;
  }
  return 0;
}


bool MLTTYDevice::isDirectAccess() const
{
  return false;
}


bool MLTTYDevice::isSequentialAccess() const
{
  return true;
}


bool MLTTYDevice::isCombinedAccess() const
{
  return false;
}


bool MLTTYDevice::isBuffered() const
{
  return false;
}


bool MLTTYDevice::isRaw() const
{
  return true;
}


bool MLTTYDevice::isSynchronous() const
{
  return true;
}


bool MLTTYDevice::isAsynchronous() const
{
  return false;
}


bool MLTTYDevice::isTranslated() const
{
  return false;
}


bool MLTTYDevice::isReadable() const
{
  if(((tty_mode&IO_ReadOnly)!=0)||((tty_mode&IO_ReadWrite)!=0)) {
    return true;
  }
  return false;
}


bool MLTTYDevice::isWritable() const
{
  if(((tty_mode&IO_WriteOnly)!=0)||((tty_mode&IO_ReadWrite)!=0)) {
    return true;
  }
  return false;
}


bool MLTTYDevice::isReadWrite() const
{
  if((tty_mode&IO_ReadWrite)!=0) {
    return true;
  }
  return false;

}


bool MLTTYDevice::isInactive() const
{
  if(!tty_open) {
    return true;
  }
  return false;
}


bool MLTTYDevice::isOpen() const
{
  if(tty_open) {
    return true;
  }
  return false;
}


int MLTTYDevice::status() const
{
  return tty_status;
}


void MLTTYDevice::resetStatus()
{
  tty_status=IO_Ok;
}


void MLTTYDevice::setName(QString name)
{
  tty_name=name;
}


int MLTTYDevice::speed() const
{
  return tty_speed;
}


void MLTTYDevice::setSpeed(int speed)
{
  tty_speed=speed;
}


int MLTTYDevice::wordLength() const
{
  return tty_length;
}


void MLTTYDevice::setWordLength(int length)
{
  tty_length=length;
}


MLTTYDevice::Parity MLTTYDevice::parity() const
{
  return tty_parity;
}


void MLTTYDevice::setParity(Parity parity)
{
  tty_parity=parity;
}


void MLTTYDevice::Init()
{
  tty_speed=9600;
  tty_length=8;
  tty_parity=MLTTYDevice::None;
  tty_open=false;
}
