// message.cpp
//
// Message Passing Widget for the Mithlond Project.
//
//   (C) Copyright 2002-2005 Fred Gleason <fredg@paravelsystems.com>
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as
//   published by the Free Software Foundation; either version 2 of
//   the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qwidget.h>
#include <qpushbutton.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qlabel.h>
#include <qpainter.h>
#include <qevent.h>
#include <qmessagebox.h>
#include <math.h>

#include <ml.h>
#include <message.h>


Message::Message(QColor base,unsigned hang_time,
		 QWidget *parent,const char *name)
		 
  : QWidget(parent,name)
{
  QFont font;

  msg_base_color=base;
  msg_hang_time=hang_time;
  msg_hanging=false;

  msg_backdrop=NULL;
  DrawBackdrop();

  //
  // Generate the Font
  //
  font=QFont("Helvetica",16,QFont::Normal);
  font.setPixelSize(16);

  //
  // The Send Box
  //
  msg_send_box=new QLineEdit(this,"msg_send_box");
  msg_send_box->setFont(font);
  msg_send_box->setMaxLength(ML_MAX_MESSAGE_LENGTH);
  connect(msg_send_box,SIGNAL(returnPressed()),this,SLOT(sendData()));

  //
  // The Receive Box
  //
  msg_rcv_box=new MsgTextEdit(this,"msg_rcv_box");
  msg_rcv_box->setFont(font);
  msg_rcv_box->setWordWrap(QTextEdit::WidgetWidth);
  msg_rcv_box->setReadOnly(true);
  connect(msg_rcv_box,SIGNAL(doubleClicked(int,int)),
	  this,SLOT(doubleClickedData(int,int)));
  connect(msg_rcv_box,SIGNAL(nicknameChanged(const QString &)),
	  this,SLOT(nicknameChangedData(const QString &)));

  //
  // The Hang Timer
  //
  msg_hang_timer=new QTimer(this,"msg_hang_timer");
  connect(msg_hang_timer,SIGNAL(timeout()),this,SLOT(timerData()));
}


QSize Message::sizeHint() const
{
  return QSize(420,183);
} 


QSizePolicy Message::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


QString Message::nickname() const
{
  return msg_nickname;
}


void Message::setNickname(const QString &name)
{
  msg_nickname=name;
  msg_rcv_box->setNickname(name);
}


void Message::setFont(const QFont &font)
{
  msg_send_box->setFont(font);
  msg_rcv_box->setFont(font);
  QWidget::setFont(font);
  setGeometry(geometry());
}


void Message::setScrollback(int lines)
{
  //  msg_rcv_box->setMaxLines(lines);
  msg_rcv_box->scrollToBottom();
}


void Message::setMessage(QString msg)
{
  msg_rcv_box->insertParagraph(msg,-1);
  msg_rcv_box->scrollToBottom();
}


QString Message::messageBuffer() const
{
  return msg_rcv_box->text();
}


void Message::setMessageBuffer(QString buffer)
{
  msg_rcv_box->clear();
  msg_rcv_box->setText(buffer);
  msg_rcv_box->scrollToBottom();
  //  msg_rcv_box->setCursorPosition(msg_rcv_box->maxLines(),0);
}


void Message::doubleClickedData(int para,int pos)
{
  if(!msg_hanging) {
    msg_rcv_box->setPaper(msg_backdrop_brush);
    msg_hang_timer->start(msg_hang_time,true);
    msg_hanging=true;
    emit touched();
  }
}


void Message::paintEvent(QPaintEvent *)
{
  QPainter *p=new QPainter(this);
  p->setPen(QColor(black));
  p->fillRect(10,10,10,10,QColor(white));
  p->end();
  msg_rcv_box->scrollToBottom();
}


int Message::numLines() const
{
  return msg_rcv_box->paragraphs();
}


QString Message::textLine(int line) const
{
  return msg_rcv_box->text(line);
}


void Message::setGeometry(int x,int y,int w,int h)
{
  QFontMetrics fm(font());

  msg_send_box->setGeometry(0,h-fm.height(),w,fm.height());
  msg_rcv_box->setGeometry(0,0,w,h-fm.height());
//  msg_send_box->setGeometry(0,h-SEND_BOX_HEIGHT,w,SEND_BOX_HEIGHT);
//  msg_rcv_box->setGeometry(0,0,w,h-SEND_BOX_HEIGHT);
  QWidget::setGeometry(x,y,w,h);
  msg_rcv_box->scrollToBottom();
  DrawBackdrop();
}


void Message::setGeometry(const QRect &rect)
{
  setGeometry(rect.x(),rect.y(),rect.width(),rect.height());
}


void Message::sendData()
{
  emit message(msg_send_box->text());
  msg_send_box->clear();
  msg_rcv_box->scrollToBottom();
}


void Message::timerData()
{
  msg_rcv_box->setPaper(msg_base_color);
  msg_hanging=false;
}


void Message::nicknameChangedData(const QString &name)
{
  msg_nickname=name;
}


void Message::DrawBackdrop()
{
  QFont font;
  int x;
  int y;

  font=QFont("Helvetica",36,QFont::Bold);
  QFontMetrics fm(font);
  x=(geometry().width()-fm.boundingRect("Look Up!").width())/2;
  y=geometry().height()/2;

  if(msg_backdrop!=NULL) {
    delete msg_backdrop;
  }
  QPainter *p=new QPainter();
  msg_backdrop=new QPixmap(geometry().width(),geometry().height());
  p->begin(msg_backdrop);
  p->setPen(QColor(yellow));
  p->fillRect(0,0,geometry().width(),geometry().height(),
	      QColor(red));
  p->setFont(font);
  p->drawText(x,y,"Look Up!");
  p->end();
  delete p;
  msg_backdrop_brush=QBrush(QColor(red),*msg_backdrop);
}
