// mithlond.cpp
//
// A Qt-based application for controling broadcast telephone interfaces.
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License as
//   published by the Free Software Foundation; either version 2 of
//   the License, or (at your option) any later version.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//


#include <qmessagebox.h>
#include <qapplication.h>
#include <qnamespace.h>
#include <qwidget.h>
#include <qpushbutton.h>
#include <qrect.h>
#include <qpoint.h>
#include <qpainter.h>
#include <qstring.h>
#include <qmessagebox.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qslider.h>
#include <qfiledialog.h>
#include <qfont.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qpointarray.h>
#include <qtimer.h>
#include <qevent.h>
#include <qaccel.h>
#include <qsignalmapper.h>
#include <qsettings.h>

#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <rconf.h>
#include <rlabel.h>
#include <rprofile.h>

#include <client_data.h>

#include <defaults.h>
#include <globals.h>
#include <mithlond.h>
#include <askbox.h>

//
// Globals
//
MlConfig *ml_config;

//
// Icons
//
#include "../icons/callcommander-22x22.xpm"


QString GetConfFile(int argc,char *argv[])
{
  QString config;
  bool file_found=false;

  //
  // Determine Which Configuration File to Use
  //
  // First, the command line argument
  //
  if(argc>1) {
    QFile *file=new QFile(QString(qApp->argv()[qApp->argc()-1]));
    if(file->exists()) {
      config=argv[argc-1];
      file_found=true;
    }
    delete file;
  }

  //
  // Next, the .mlrc File
  //
  if(!file_found) {
    bool found;
    QString home=RGetHomeDir(&found);
    if(found) {
      if(QFile(QString().sprintf("%s/.mlrc",(const char *)home)).exists()) {
	config=QString().sprintf("%s/.mlrc",(const char *)home);
	file_found=true;
      }
    }
  }

  //
  // Finally, the global configuration
  //
#ifdef WIN32
  if(!file_found) {
    QSettings settings=QSettings();
    settings.insertSearchPath(QSettings::Windows,ML_REGISTRY_PATH);
    config=
      QString().sprintf("%s\\ml.ini",
			(const char *)settings.
			readEntry(ML_REGISTRY_INSTALL_DIR,"C:\\etc"));
  }
#else
  if(!file_found) {
    config=DEFAULT_CONFIG_FILE;
  }
#endif  // WIN32

  return config;
}


MainWidget::MainWidget(QWidget *parent,const char *name,WFlags f)
  :QMainWindow(parent,name,f)
{
  QFont font;
  QString label;
  telos_metadata_timer=NULL;
  telos_connection=new ConnectionData(ConnectionData::StorageFilesystem);

  //
  // Create And Set Icon
  //
  telos_callcommander_map=new QPixmap(callcommander_xpm);
  setIcon(*telos_callcommander_map);

  //
  // The Line Warning Dialogs
  //
  for(int i=0;i<MAX_LINES;i++) {
    telos_warning_box[i]=new LineWarning(this,"telos_warning_box");
  }

  LoadConfig();
  DrawCaps();
  telos_screen_state=false;
  telos_mute_state=false;
  telos_conf_state=false;
  telos_vip_state=false;
  telos_record_frame=0;
  telos_record_running=false;
  telos_director_mode=ConsoleData::Talent;

  setBackgroundColor(QColor(black));
  SetCaption();
  if(ml_config->keepOnTop()) {
    setWFlags(Qt::WStyle_StaysOnTop);
  }

  //
  // The Bus Driver
  //
  switch(telos_connection->systemType()) {
      case ConnectionData::TypeTelos2101:
	telos_driver=new Telos2101Driver(0,this,"telos_driver");
	break;

      case ConnectionData::TypeTelos100Key:
	telos_driver=
	  new Telos100Driver(Telos100Driver::KeyType,0,this,"telos_driver");
	break;

      case ConnectionData::TypeTelos100Dim:
	telos_driver=
	  new Telos100Driver(Telos100Driver::DimType,0,this,"telos_driver");
	break;

      case ConnectionData::TypeTelos100OneXSix:
	telos_driver=
	  new Telos100Driver(Telos100Driver::OneXSixType,0,this,
			     "telos_driver");
	break;

      case ConnectionData::TypeGentnerTs612:
	telos_driver=new Ts612Driver(0,this,"telos_driver");
	break;

      case ConnectionData::TypeVirtual:
	telos_driver=new VirtualDriver(0,this,"telos_driver");
	break;

      case ConnectionData::TypeUnknown:
	QMessageBox::warning(this,tr("System Error"),
			     tr("Unknown System Type!"));
	exit(256);
  }
  telos_driver->setClientType(ClientData::TypeUser);
  connect(telos_driver,SIGNAL(connected(unsigned,bool)),
	  this,SLOT(connectedData(unsigned,bool)));
  connect(telos_driver,SIGNAL(connectionClosed(unsigned)),
	  this,SLOT(connectionClosedData(unsigned)));
  connect(telos_driver,SIGNAL(currentShow(unsigned,const QString &,bool)),
	  this,SLOT(currentShowData(unsigned,const QString &,bool)));
  connect(telos_driver,SIGNAL(currentConsole(unsigned,int,ConsoleData::Mode)),
	  this,SLOT(currentConsoleData(unsigned,int,ConsoleData::Mode)));
					     
  connect(telos_driver,SIGNAL(error(unsigned,BusDriver::Error)),
	  this,SLOT(errorData(unsigned,BusDriver::Error)));
  connect(telos_driver,SIGNAL(currentLineState(unsigned,BusDriver::LineState,
					       unsigned,int,int)),this,
	  SLOT(lineStateData(unsigned,BusDriver::LineState,unsigned,int,int)));
  connect(telos_driver,SIGNAL(currentBusyAllState(unsigned,bool)),
	  this,SLOT(busyAllStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentDumpDelayState(unsigned,bool)),
	  this,SLOT(dumpDelayStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentRecordState(unsigned,bool)),
	  this,SLOT(recordStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentVipState(unsigned,bool)),
	  this,SLOT(vipStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentScreenState(unsigned,bool)),
	  this,SLOT(screenStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentConferenceState(unsigned,bool)),
	  this,SLOT(conferenceStateData(unsigned,bool)));
  connect(telos_driver,SIGNAL(currentMuteState(unsigned,bool)),
	  this,SLOT(muteStateData(unsigned,bool)));
  connect(telos_driver,
	  SIGNAL(receivedString(unsigned,BusDriver::Element,unsigned,
				const QString &)),
	  this,SLOT(receivedStringData(unsigned,BusDriver::Element,unsigned,
				       const QString &)));
  connect(telos_driver,
	  SIGNAL(metadataUpdateRequested(unsigned,ClientData::ClientType)),
	  this,
	  SLOT(metadataUpdateRequestedData(unsigned,ClientData::ClientType)));

  //
  // The List Clients Dialog
  //
  telos_list_clients=new ListClients(telos_driver,this);

  //
  // Line Buttons
  //
  QSignalMapper *mapper=new QSignalMapper(this,"mapper");
  connect(mapper,SIGNAL(mapped(int)),this,SLOT(lineButtonData(int)));
  for(int i=0;i<MAX_BANKS;i++) {
    for(int j=0;j<MAX_LINES;j++) {
      telos_line[i][j]=NULL;
      if(ml_config->lineShow(i,j)) {
	telos_line[i][j]=new QPushButton(this,"telos_line");
	telos_line[i][j]->setGeometry(ml_config->linePosition(i,j).x(),
				      ml_config->linePosition(i,j).y(),80,80);
	telos_line[i][j]->setAccel(QKeySequence(ml_config->lineKey(i,j)));
	telos_line[i][j]->setPalette(ml_config->palette(MlConfig::IdleColor));
	mapper->setMapping(telos_line[i][j],MAX_LINES*i+j);
	if(!ml_config->disableLineButtons()) {
	  connect(telos_line[i][j],SIGNAL(clicked()),mapper,SLOT(map()));
	}
      }
      else {
	telos_line[i][j]=NULL;
      }
    }
  }

  //
  // Hold Buttons
  //
  if(ml_config->bankShowHoldButton(0)) {
    telos_hold_button[0]=new QPushButton(this,"telos_hold");
    telos_hold_button[0]->
      setGeometry(ml_config->bankHoldPosition(0).x(),
		  ml_config->bankHoldPosition(0).y(),80,80);
    telos_hold_button[0]->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_hold_button[0]->setPixmap(telos_hold_icon);
    telos_hold_button[0]->setAccel(QKeySequence(ml_config->bankHoldKey(0)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_hold_button[0],SIGNAL(clicked()),
	      this,SLOT(mainHoldData()));
    }
  }
  if(ml_config->bankShowHoldButton(1)) {
    telos_hold_button[1]=new QPushButton(this,"telos_hold");
    telos_hold_button[1]->
      setGeometry(ml_config->bankHoldPosition(1).x(),
		  ml_config->bankHoldPosition(1).y(),80,80);
    telos_hold_button[1]->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_hold_button[1]->setPixmap(telos_hold_icon);
    telos_hold_button[1]->setAccel(QKeySequence(ml_config->bankHoldKey(1)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_hold_button[1],SIGNAL(clicked()),
	      this,SLOT(confHoldData()));
    }
  }

  //
  // Drop Buttons
  //
  if(ml_config->bankShowDropButton(0)) {
    telos_drop_button[0]=new QPushButton(this,"telos_drop");
    telos_drop_button[0]->
      setGeometry(ml_config->bankDropPosition(0).x(),
		  ml_config->bankDropPosition(0).y(),80,80);
    telos_drop_button[0]->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_drop_button[0]->setPixmap(telos_drop_icon);
    telos_drop_button[0]->setAccel(QKeySequence(ml_config->bankDropKey(0)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_drop_button[0],SIGNAL(clicked()),
	      this,SLOT(mainDropData()));
    }
  }
  if(ml_config->bankShowDropButton(1)) {
    telos_drop_button[1]=new QPushButton(this,"telos_drop");
    telos_drop_button[1]->
      setGeometry(ml_config->bankDropPosition(1).x(),
		  ml_config->bankDropPosition(1).y(),80,80);
    telos_drop_button[1]->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_drop_button[1]->setPixmap(telos_drop_icon);
    telos_drop_button[1]->setAccel(QKeySequence(ml_config->bankDropKey(1)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_drop_button[1],SIGNAL(clicked()),
	      this,SLOT(confDropData()));
    }
  }

  //
  // Caller Data Boxes
  //
  QAccel *accel=new QAccel(this,"accel");
  if(!ml_config->disableLineBoxes()) {
    connect(accel,SIGNAL(activated(int)),this,SLOT(callerBoxData(int)));
  }
  for(int i=0;i<MAX_LINES;i++) {
    telos_caller_data[i]=new CallMetaData();
    if(ml_config->lineShowCallerBox(i)) {
      if(ml_config->callerShow(MlConfig::CallerName)) {
	telos_caller_name_label[i]=new RLabel(this,"name_label");
	telos_caller_name_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerName).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerName).y(),
		      ml_config->callerWindow(MlConfig::CallerName).width(),
		      ml_config->callerWindow(MlConfig::CallerName).height());
	telos_caller_name_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerName));
	telos_caller_name_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerAge)) {
	telos_caller_age_label[i]=new RLabel(this,"age_label");
	telos_caller_age_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerAge).x(),
		      ml_config->lineCallerBoxWindow(i).y()+	
		      ml_config->callerWindow(MlConfig::CallerAge).y(),
		      ml_config->callerWindow(MlConfig::CallerAge).width(),
		      ml_config->callerWindow(MlConfig::CallerAge).height());
	telos_caller_age_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerAge));
	telos_caller_age_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerCityState)) {
	telos_caller_citystate_label[i]=new RLabel(this,"citystate_label");
	telos_caller_citystate_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerCityState).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerCityState).y(),
		      ml_config->
		      callerWindow(MlConfig::CallerCityState).width(),
		      ml_config->
		      callerWindow(MlConfig::CallerCityState).height());
	telos_caller_citystate_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerCityState));
	telos_caller_citystate_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerZip)) {
	telos_caller_zip_label[i]=new RLabel(this,"zip_label");
	telos_caller_zip_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerZip).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerZip).y(),
		      ml_config->callerWindow(MlConfig::CallerZip).width(),
		      ml_config->callerWindow(MlConfig::CallerZip).height());
	telos_caller_zip_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerZip));
	telos_caller_zip_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerStation)) {
	telos_caller_station_label[i]=new RLabel(this,"station_label");
	telos_caller_station_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerStation).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerStation).y(),
		      ml_config->callerWindow(MlConfig::CallerStation).width(),
		      ml_config->callerWindow(MlConfig::CallerStation).height());
	telos_caller_station_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerStation));
	telos_caller_station_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerGender)) {
	telos_caller_gender_label[i]=new RLabel(this,"gender_label");
	telos_caller_gender_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerGender).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerGender).y(),
		      ml_config->callerWindow(MlConfig::CallerGender).width(),
		      ml_config->
		      callerWindow(MlConfig::CallerGender).height());
	telos_caller_gender_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerGender));
	telos_caller_gender_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }
      if(ml_config->callerShow(MlConfig::CallerPhone)) {
	telos_caller_phone_label[i]=new RLabel(this,"phone_label");
	telos_caller_phone_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerPhone).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerPhone).y(),
		      ml_config->callerWindow(MlConfig::CallerPhone).width(),
		      ml_config->callerWindow(MlConfig::CallerPhone).height());
	telos_caller_phone_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerPhone));
	telos_caller_phone_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }

      if(ml_config->callerShow(MlConfig::CallerCallCounter)) {
	telos_caller_callcounter_label[i]=new RLabel(this,"phone_label");
	telos_caller_callcounter_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerCallCounter).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerCallCounter).y(),
		      ml_config->
		      callerWindow(MlConfig::CallerCallCounter).width(),
		      ml_config->
		      callerWindow(MlConfig::CallerCallCounter).height());
	telos_caller_callcounter_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerCallCounter));
	telos_caller_callcounter_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
      }

      if(ml_config->callerShow(MlConfig::CallerComment)) {
	telos_caller_comment_label[i]=new RLabel(this,"comment_label");
	telos_caller_comment_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerComment).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerComment).y(),
		      ml_config->callerWindow(MlConfig::CallerComment).width(),
		      ml_config->
		      callerWindow(MlConfig::CallerComment).height());
	telos_caller_comment_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
	telos_caller_comment_label[i]->setAlignment(AlignTop);
	telos_caller_comment_label[i]->
	  setFont(ml_config->callerFont(MlConfig::CallerComment));
	telos_caller_comment_label[i]->setWordWrapEnabled(true);
      }

      if(ml_config->callerShow(MlConfig::CallerCell)) {
	telos_caller_cell_label[i]=new QLabel(this,"cell_label");
	telos_caller_cell_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerCell).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerCell).y(),
		      25,25);
	telos_caller_cell_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
	telos_caller_cell_label[i]->setAlignment(AlignTop);
      }
      if(ml_config->callerShow(MlConfig::CallerQuality)) {
	telos_caller_quality_label[i]=new QLabel(this,"quality_label");
	telos_caller_quality_label[i]->
	  setGeometry(ml_config->lineCallerBoxWindow(i).x()+
		      ml_config->callerWindow(MlConfig::CallerQuality).x(),
		      ml_config->lineCallerBoxWindow(i).y()+
		      ml_config->callerWindow(MlConfig::CallerQuality).y(),
		      25,25);
	telos_caller_quality_label[i]->
	  setPalette(ml_config->color(MlConfig::CallerBoxColor));
	telos_caller_quality_label[i]->setAlignment(AlignTop);
      }
//      UpdateCallerBox(i);
    }
    accel->insertItem(ml_config->lineCallerKey(i),i);
  }
  telos_caller_check_data=new CallMetaData();

  //
  // The Message Box
  //
  if(ml_config->msgboxShow()) {
    telos_message_edit=new Message(ml_config->color(MlConfig::MessageBoxColor),
				   ml_config->msgboxHangTime(),
				   this,"telos_message_edit");
    telos_message_edit->setGeometry(ml_config->msgboxWindow());
    telos_message_edit->
      setPalette(ml_config->color(MlConfig::MessageBoxColor));
    telos_message_edit->setFont(ml_config->msgboxFont());
    telos_message_edit->setScrollback(ml_config->msgboxScrollbackLines());
    connect(telos_message_edit,SIGNAL(message(QString)),
	    this,SLOT(messageData(QString)));
    connect(telos_message_edit,SIGNAL(touched()),this,SLOT(touchedData()));
    telos_message_data=new MessageData();
    telos_message_id="";
  }

  //
  // Clock Widget
  //
  if(ml_config->clockShowClock()) {
    telos_clock_segclock=new RSegClock(ml_config->clockOrientation(),this,
				       "telos_clock_segclock");
    if(ml_config->clockOrientation()==Qt::Horizontal) {
      telos_clock_segclock->
	setGeometry(ml_config->clockPosition().x(),
		    ml_config->clockPosition().y(),347,95);
    }
    else {
      telos_clock_segclock->setGeometry(ml_config->clockPosition().x(),
					ml_config->clockPosition().y(),
					172,170);
    }
    telos_clock_segclock->setOffset(ml_config->clockTimeOffset());
    telos_clock_segclock->loadSegList(ml_config->segmentList());
    telos_clock_segclock->setShowHour(ml_config->clockShowHour());
    if(ml_config->clockMilitaryFormat()) {
      telos_clock_segclock->set24Hour();
    }
    else {
      telos_clock_segclock->set12Hour();
    }
  }

  //
  // The Busy Button
  //
  if(ml_config->buttonShow(MlConfig::BusyButton)) {
    telos_busy_button=new QPushButton(this,"telos_busy_button");
    telos_busy_button->setGeometry(ml_config->
				   buttonPosition(MlConfig::BusyButton).x(),
				   ml_config->
				   buttonPosition(MlConfig::BusyButton).y(),
				   80,80);
    telos_busy_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_busy_button->setPixmap(telos_busy_icon);
    telos_busy_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::BusyButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_busy_button,SIGNAL(clicked()),this,SLOT(busyData()));
    }
  }

  //
  // The Dump Button
  //
  if(ml_config->buttonShow(MlConfig::DumpButton)) {
    telos_dump_button=new QPushButton(this,"telos_dump_button");
    telos_dump_button->
      setGeometry(ml_config->buttonPosition(MlConfig::DumpButton).x(),
		  ml_config->buttonPosition(MlConfig::DumpButton).y(),80,80);
    telos_dump_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_dump_button->setPixmap(telos_dump_icon);
    telos_dump_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::DumpButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_dump_button,SIGNAL(pressed()),
	      this,SLOT(dumpPressedData()));
      connect(telos_dump_button,SIGNAL(released()),
	      this,SLOT(dumpReleasedData()));
    }
  }

  //
  // The Record Button
  //
  if(ml_config->buttonShow(MlConfig::RecordButton)) {
    telos_record_button=new QPushButton(this,"telos_record_button");
    telos_record_button->
      setGeometry(ml_config->buttonPosition(MlConfig::RecordButton).x(),
		  ml_config->buttonPosition(MlConfig::RecordButton).y(),80,80);
    telos_record_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_record_button->setPixmap(telos_record_icon[0]);
    telos_record_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::RecordButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_record_button,SIGNAL(clicked()),this,SLOT(recordData()));
    }
  }

  //
  // The Next Button
  //
  if(ml_config->buttonShow(MlConfig::NextButton)) {
    telos_next_button=new QPushButton(this,"telos_next_button");
    telos_next_button->
      setGeometry(ml_config->buttonPosition(MlConfig::NextButton).x(),
		  ml_config->buttonPosition(MlConfig::NextButton).y(),80,80);
    telos_next_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_next_button->setPixmap(telos_next_icon);
    telos_next_button->
      setPalette(ml_config->palette(MlConfig::NextButtonColor));
    telos_next_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::NextButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_next_button,SIGNAL(clicked()),this,SLOT(nextData()));
    }
  }

  //
  // The Keypad Button
  //
  if(ml_config->buttonShow(MlConfig::KeypadButton)) {
    telos_keypad_button=new QPushButton(this,"telos_keypad_button");
    telos_keypad_button->
      setGeometry(ml_config->buttonPosition(MlConfig::KeypadButton).x(),
		  ml_config->buttonPosition(MlConfig::KeypadButton).y(),80,80);
    telos_keypad_button->setPixmap(telos_keypad_icon);
    telos_keypad_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_keypad_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::KeypadButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_keypad_button,SIGNAL(clicked()),this,SLOT(keypadData()));
    }
  }

  //
  // Generate the font for Gentner Buttons
  //
  font=QFont("Helvetica",18,QFont::Bold);
  font.setPixelSize(18);

  //
  // The Flash Button
  //
  if(ml_config->buttonShow(MlConfig::FlashButton)) {
    telos_flash_button=new QPushButton(this,"telos_flash_button");
    telos_flash_button->
      setGeometry(ml_config->buttonPosition(MlConfig::FlashButton).x(),
		  ml_config->buttonPosition(MlConfig::FlashButton).y(),80,80);
    telos_flash_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_flash_button->setFont(font);
    telos_flash_button->setText("FLSH");
    telos_flash_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::FlashButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_flash_button,SIGNAL(clicked()),this,SLOT(flashData()));
    }
  }

  //
  // The Screen Button
  //
  if(ml_config->buttonShow(MlConfig::ScreenButton)) {
    telos_screen_button=new QPushButton(this,"telos_screen_button");
    telos_screen_button->
      setGeometry(ml_config->buttonPosition(MlConfig::ScreenButton).x(),
		  ml_config->buttonPosition(MlConfig::ScreenButton).y(),80,80);
    telos_screen_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_screen_button->setFont(font);
    telos_screen_button->setText("SCRN");
    telos_screen_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::ScreenButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_screen_button,SIGNAL(clicked()),this,SLOT(screenData()));
    }
  }

  //
  // The Mute Button
  //
  if(ml_config->buttonShow(MlConfig::MuteButton)) {
    telos_mute_button=new QPushButton(this,"telos_mute_button");
    telos_mute_button->
      setGeometry(ml_config->buttonPosition(MlConfig::MuteButton).x(),
		  ml_config->buttonPosition(MlConfig::MuteButton).y(),80,80);
    telos_mute_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_mute_button->setFont(font);
    telos_mute_button->setText("MUTE");
    telos_mute_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::MuteButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_mute_button,SIGNAL(clicked()),this,SLOT(muteData()));
    }
  }

  //
  // The Conf Button
  //
  if(ml_config->buttonShow(MlConfig::ConfButton)) {
    telos_conf_button=new QPushButton(this,"telos_conf_button");
    telos_conf_button->
      setGeometry(ml_config->buttonPosition(MlConfig::ConfButton).x(),
		  ml_config->buttonPosition(MlConfig::ConfButton).y(),80,80);
    telos_conf_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_conf_button->setFont(font);
    telos_conf_button->setText("CONF");
    telos_conf_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::ConfButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_conf_button,SIGNAL(clicked()),this,SLOT(confData()));
    }
  }

  //
  // The Vip Button
  //
  if(ml_config->buttonShow(MlConfig::VipButton)) {
    telos_vip_button=new QPushButton(this,"telos_vip_button");
    telos_vip_button->
      setGeometry(ml_config->buttonPosition(MlConfig::VipButton).x(),
		  ml_config->buttonPosition(MlConfig::VipButton).y(),80,80);
    telos_vip_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_vip_button->setFont(font);
    telos_vip_button->setText("VIP");
    telos_vip_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::VipButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_vip_button,SIGNAL(clicked()),this,SLOT(vipData()));
    }
  }

  //
  // The Client Button
  //
  if(ml_config->buttonShow(MlConfig::ClientButton)) {
    telos_client_button=new QPushButton(this,"telos_client_button");
    telos_client_button->
      setGeometry(ml_config->buttonPosition(MlConfig::ClientButton).x(),
		  ml_config->buttonPosition(MlConfig::ClientButton).y(),80,80);
    telos_client_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    telos_client_button->setFont(font);
    telos_client_button->setText("Who's\nHere?");
    telos_client_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::ClientButton)));
    if(!ml_config->disableLineButtons()) {
      connect(telos_client_button,SIGNAL(clicked()),this,SLOT(clientData()));
    }
  }

  //
  // Initialize Line Status
  //
  for(int i=0;i<MAX_LINES;i++) {
    telos_status[i]=BusDriver::LineInactive;
  }

  setGeometry(ml_config->window());

  //
  // Initlize Logging
  //
  if(!InitLogging()) {
    QMessageBox::warning(this,"Log Failure",
			 "Unable to initialize logging system");
    qApp->quit();
  }

  //
  // Start the Connection
  //
  switch(telos_connection->systemType()) {
      case ConnectionData::TypeTelos2101:
      case ConnectionData::TypeVirtual:
	telos_driver->connectToHost(telos_connection->hostName(),
				    telos_connection->userName(),
				    telos_connection->userPassword(),
				    telos_connection->tcpPort());
	break;

      case ConnectionData::TypeTelos100Key:
      case ConnectionData::TypeTelos100Dim:
      case ConnectionData::TypeTelos100OneXSix:
      case ConnectionData::TypeGentnerTs612:
	telos_driver->connectToSerial(telos_connection->ttyPort());
	break;

      case BusDriver::UnknownConnection:
	QMessageBox::warning(this,tr("System Error"),
			     tr("Unknown System Type!"));
	exit(256);
  }
}


QSize MainWidget::sizeHint() const
{
  return ml_config->window().size();
}


QSizePolicy MainWidget::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


void MainWidget::closeEvent(QCloseEvent *e)
{
  telos_driver->disconnect();
  qApp->processEvents();
  exit(0);
}


void MainWidget::paintEvent(QPaintEvent *paintevent)
{
  //
  // Caller Box Background
  //
  QPainter *p=new QPainter(this);
  for(int i=0;i<MAX_LINES;i++) {
    if(ml_config->lineShowCallerBox(i)) {
      if(paintevent->rect().intersects(ml_config->lineCallerBoxWindow(i))) {
	p->setPen(QColor(black));
	p->fillRect(ml_config->lineCallerBoxWindow(i).x(),
		    ml_config->lineCallerBoxWindow(i).y(),
		    ml_config->callerSize().width(),
		    ml_config->callerSize().height(),
		    ml_config->color(MlConfig::CallerBoxColor));
	p->drawRect(ml_config->lineCallerBoxWindow(i).x(),
		    ml_config->lineCallerBoxWindow(i).y(),
		    ml_config->callerSize().width(),
		    ml_config->callerSize().height());
      }
    }
  }
  p->end();

}


void MainWidget::moveEvent(QMoveEvent *e)
{
  for(int i=0;i<MAX_LINES;i++) {
    if(telos_warning_box[i]->isVisible()) {
      telos_warning_box[i]->
	setGeometry(ml_config->lineCallerBoxWindow(i).x()+e->pos().x()+
		    frameGeometry().x()-geometry().x(),
		    ml_config->lineCallerBoxWindow(i).y()+e->pos().y()+
		    frameGeometry().y()-geometry().y(),
		    ml_config->callerSize().width(),
		    ml_config->callerSize().height());
    }
  }
  QMainWindow::moveEvent(e);
}


void MainWidget::errorData(unsigned id,BusDriver::Error err)
{
  switch(err) {
      case BusDriver::NoError:
	break;

      case BusDriver::HostNotFound:
	break;

      case BusDriver::ConnectionRefused:
	QMessageBox::warning(this,"Connection Refused",
			     "The connection was refused by the server!");
	qApp->quit();
	break;

      case BusDriver::ReadError:
	QMessageBox::warning(this,"Read Error","Socket Read Error!");
	break;

      default:
	QMessageBox::warning(this,"Unknown Error",
			     "Unrecoginized Socket Error!");
	break;
  }
}


void MainWidget::lineButtonData(int id)
{
  ProcessLineButton(id);
}


void MainWidget::messageData(QString msg)
{
  if(ml_config->msgboxShow()) {
    telos_message_edit->
      setMessage(ml_config->msgboxNickname()+QString("> ")+msg);
    telos_message_data->clear();
    telos_message_data->setNick(ml_config->msgboxNickname());
    telos_message_data->setMessage(msg);
    telos_message_data->sendData(telos_driver);
  }
}


void MainWidget::touchedData()
{
  telos_driver->sendString(BusDriver::LookupGlobalElement,0,"lookup");
}


void MainWidget::mouseDoubleClickEvent(QMouseEvent *e)
{
  if(!ml_config->disableLineBoxes()) {
    for(int i=0;i<MAX_LINES;i++) {
      if(ml_config->lineShowCallerBox(i)) {
	if((e->x()>ml_config->lineCallerBoxWindow(i).x())&&
	   (e->x()<ml_config->lineCallerBoxWindow(i).x()+
	    ml_config->callerSize().width())&&
	   (e->y()>ml_config->lineCallerBoxWindow(i).y())&&
	   (e->y()<ml_config->lineCallerBoxWindow(i).y()+
	    ml_config->callerSize().height()))
	{
	  callerBoxData(i);
	}
      }
    }
  }
}


void MainWidget::callerBoxData(int line)
{
  int dialog_exit;

  if(telos_status[line]!=BusDriver::LineInactive) {
    telos_caller_dialog=new Callers(false,telos_caller_data[line],
				    this,"caller_dialog");
    dialog_exit=telos_caller_dialog->exec();
    if(dialog_exit!=1) {
      UpdateCallerBox(line);
      SendCallData(line);
    }
    switch(telos_status[line]) {
	case BusDriver::LineHold:
	case BusDriver::LineScreenedHold:
	case BusDriver::LineNext:
	  if(telos_status[line]!=BusDriver::LineOnAir) {
	    UpdateTimer(line,telos_caller_data[line]->
			element(BusDriver::HoldTimerElement).toInt());
	  }
	  break;
	  
	default:
	  break;
    }
    delete telos_caller_dialog;
    telos_caller_dialog=NULL;
    repaint(ml_config->lineCallerBoxWindow(line).x(),
	    ml_config->lineCallerBoxWindow(line).y(),
	    ml_config->callerSize().width(),
	    ml_config->callerSize().height());
  }
}


void MainWidget::ProcessLineButton(int line)
{
  int real_line=line;

  if(line<12) {
    real_line=line;
  }
  else {
    real_line=line-12;
  }
  switch(telos_status[real_line]) {
      case BusDriver::LineLocked:
	if(line<12) {
	  if(telos_bank[real_line]==0) {
	    telos_driver->selectLine(line,telos_connection->console(),0);
	  }
	}
	else {
	  if(telos_bank[real_line]==1) {
	    telos_driver->selectLine(line-12,telos_connection->console(),1);
	  }
	}
	break;
	
      case BusDriver::LineIdle:
      case BusDriver::LineOnAir:
      case BusDriver::LineHold:
      case BusDriver::LineScreenedHold:
      case BusDriver::LineNext:
      case BusDriver::LineRinging:
	if(line<12) {
	  telos_driver->selectLine(line,telos_connection->console(),0);
	}
	else {
	  telos_driver->selectLine(line-12,telos_connection->console(),1);
	}
	break;
	
      default:
	break;
  }
}


void MainWidget::mainHoldData()
{
  telos_driver->hold(0);
}
 

void MainWidget::mainDropData()
{
  telos_driver->drop(0);
}


void MainWidget::confHoldData()
{
  telos_driver->hold(1);
}


void MainWidget::confDropData()
{
  telos_driver->drop(1);
}


void MainWidget::connectedData(unsigned id,bool state)
{
  if(state) {
    telos_driver->connectToShow(telos_connection->showName(),
				telos_connection->showPassword(),
				telos_connection->location());
  }
  else {
    QMessageBox::warning(this,tr("Error"),
			 tr("Invalid Login!\nUnable to connect."),
			 1,0,0);
    qApp->quit();
  }
}


void MainWidget::connectionClosedData(unsigned id)
{
}


void MainWidget::currentShowData(unsigned id,const QString &show_name,
				 bool state)
{
  if(state) {
    telos_driver->connectToConsole(telos_connection->console(),
				   telos_connection->consoleName());
    if(!ml_config->disableStatusSync()) {
      telos_driver->requestLineState();
    }
  }
  else {
    QMessageBox::warning(this,tr("Error"),
	   tr("The specified Show does not exist!\nUnable to connect."));
    exit(256);
  }
}


void MainWidget::currentConsoleData(unsigned id,int console,
				    ConsoleData::Mode mode)
{
  if(console!=telos_connection->console()) {
    return;
  }
  if(mode==ConsoleData::Unavailable) {
    QMessageBox::warning(this,tr("Invalid Console"),
			 tr("The specified Console doesn't exist!"));
    exit(256);
  }
  telos_director_mode=mode;
  SetCaption();

  if(telos_metadata_timer==NULL) {
    telos_metadata_timer=new QTimer(this,"telos_metadata_timer");
    connect(telos_metadata_timer,SIGNAL(timeout()),
	    this,SLOT(requestMetadataData()));
    telos_metadata_timer->start(REQUEST_METADATA_DELAY,true);
  }
}


void MainWidget::lineStateData(unsigned id,BusDriver::LineState lstate,
			       unsigned line,int bank,int console)
{
//  printf("lineStateData:  lstate: %d  line: %d  bank: %d  console: %d\n",lstate,line,bank,console);
  if((telos_status[line]==lstate)&&(bank==telos_bank[line])) {
    return;
  }
  telos_status[line]=lstate;
  telos_bank[line]=bank;
  switch(lstate) {
      case BusDriver::LineIdle:
	LogCall(line);
	telos_caller_data[line]->clear();
	ClearCallerBox(line);
	update(ml_config->lineCallerBoxWindow(line).x(),
	       ml_config->lineCallerBoxWindow(line).y(),
	       ml_config->callerSize().width(),
	       ml_config->callerSize().height());
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(QPalette(ml_config->
				palette(MlConfig::IdleColor)));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][0][telos_icon_frame[0]]);
	  telos_line[0][line]->setEnabled(true);	      
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(QPalette(ml_config->
				palette(MlConfig::IdleColor)));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][0][telos_icon_frame[0]]);
	  telos_line[1][line]->setEnabled(true);	      
	}
	telos_caller_data[line]->setElement(BusDriver::AirTimerElement,0);
	telos_caller_data[line]->setElement(BusDriver::HoldTimerElement,0);
	telos_caller_data[line]->setElement(BusDriver::RingTimerElement,0);
	telos_warning_box[line]->hide();
	break;
	
      case BusDriver::LineInactive:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][10][telos_icon_frame[10]]);
	  telos_line[0][line]->setDisabled(true);	      
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][10][telos_icon_frame[10]]);
	  telos_line[1][line]->setDisabled(true);	      
	}
	break;
	
      case BusDriver::LineElsewhere:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(QPalette(ml_config->
				palette(MlConfig::OffHookColor)));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(QPalette(ml_config->
				palette(MlConfig::OffHookColor)));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	}
	break;
	
      case BusDriver::LineOnAir:
	if(telos_line[0][line]!=NULL) {
	  if(bank==0) {
	    telos_line[0][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OnAirColor)));
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][1][telos_icon_frame[1]]);
	  }
	  else {
	    telos_line[0][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OffHookColor)));
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[1]]);
	  }
	}
	if(telos_line[1][line]!=NULL) {
	  if(bank==0) {
	    telos_line[1][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OffHookColor)));
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	  else {
	    telos_line[1][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OnAirColor)));
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][2][telos_icon_frame[4]]);
	  }
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::AirTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineLocked:
	if(telos_line[0][line]!=NULL) {
	  if(bank==0) {
	    telos_line[0][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OnAirColor)));
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][3][telos_icon_frame[3]]);
	  }
	  else {
	    telos_line[0][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OffHookColor)));
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(telos_line[1][line]!=NULL) {
	  if(bank==1) {
	    telos_line[1][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OnAirColor)));
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][3][telos_icon_frame[3]]);
	  }
	  else {
	    telos_line[1][line]->
	      setPalette(QPalette(ml_config->
				  palette(MlConfig::OffHookColor)));
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::AirTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineSpeaker:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::OffHookColor));
	  if(telos_connection->console()==console) {
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][12][telos_icon_frame[12]]);
	  }
	  else {
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::OffHookColor));
	  if(telos_connection->console()==console) {
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][12][telos_icon_frame[12]]);
	  }
	  else {
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::HoldTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineHandset:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::OffHookColor));
	  if(telos_connection->console()==console) {
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][11][telos_icon_frame[11]]);
	  }
	  else {
	    telos_line[0][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::OffHookColor));
	  if(telos_connection->console()==console) {
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][11][telos_icon_frame[11]]);
	  }
	  else {
	    telos_line[1][line]->
	      setPixmap(telos_cap[line][4][telos_icon_frame[4]]);
	  }
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::HoldTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineHold:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::OnHoldColor));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][5][telos_icon_frame[5]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::OnHoldColor));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][5][telos_icon_frame[5]]);
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::HoldTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineScreenedHold:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::ScreenedHoldColor));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][6][telos_icon_frame[6]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::ScreenedHoldColor));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][6][telos_icon_frame[6]]);
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::HoldTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineNext:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::ReadyNextColor));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][7][telos_icon_frame[7]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::ReadyNextColor));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][7][telos_icon_frame[7]]);
	}
	if(ml_config->lineShowCallerBox(line)) {
	  UpdateTimer(line,telos_caller_data[line]->
		      element(BusDriver::HoldTimerElement).toInt());
	}
	break;
	
      case BusDriver::LineRinging:
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::RingingColor));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][8][telos_icon_frame[8]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::RingingColor));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][8][telos_icon_frame[8]]);
	}
	break;
	
      case BusDriver::LineBusied:
	telos_caller_data[line]->clear();
	ClearCallerBox(line);
	update(ml_config->lineCallerBoxWindow(line).x(),
	       ml_config->lineCallerBoxWindow(line).y(),
	       ml_config->callerSize().width(),
	       ml_config->callerSize().height());
	if(telos_line[0][line]!=NULL) {
	  telos_line[0][line]->
	    setPalette(ml_config->palette(MlConfig::BusyColor));
	  telos_line[0][line]->
	    setPixmap(telos_cap[line][9][telos_icon_frame[9]]);
	}
	if(telos_line[1][line]!=NULL) {
	  telos_line[1][line]->
	    setPalette(ml_config->palette(MlConfig::BusyColor));
	  telos_line[1][line]->
	    setPixmap(telos_cap[line][9][telos_icon_frame[9]]);
	}
	break;
  }
  if(ml_config->lineShow(0,line)) {
    telos_line[0][line]->
      setAccel(QKeySequence(ml_config->lineKey(0,line)));
  }
  if(telos_line[1][line]) {
    telos_line[1][line]->
      setAccel(QKeySequence(ml_config->lineKey(1,line)));
  }
}


void MainWidget::busyAllStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::BusyButton)) {
      telos_busy_button->
	setPalette(ml_config->palette(MlConfig::BusyColor));
      telos_busy_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::BusyButton)));
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::BusyButton)) {
      telos_busy_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
      telos_busy_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::BusyButton)));
    }
  }
}


void MainWidget::dumpDelayStateData(unsigned id,bool state)
{
  if(ml_config->buttonShow(MlConfig::DumpButton)) {
    if(state) {
      telos_dump_button->
	setPalette(ml_config->palette(MlConfig::DumpButtonColor));
    }
    else {
      telos_dump_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
    }
    telos_dump_button->
      setAccel(QKeySequence(ml_config->buttonKey(MlConfig::DumpButton)));
  }
}


void MainWidget::recordStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::RecordButton)) {
      telos_record_running=true;
      telos_record_button->setPalette(ml_config->palette(MlConfig::RecordButtonColor));
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::RecordButton)) {
      telos_record_running=false;
      telos_record_button->setPalette(ml_config->palette(MlConfig::IdleColor));
    }
  }
}


void MainWidget::screenStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::ScreenButton)) {
      telos_screen_button->
	setPalette(ml_config->palette(MlConfig::ScreenButtonColor));
      telos_screen_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::ScreenButton)));
      telos_screen_state=true;
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::ScreenButton)) {
      telos_screen_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
      telos_screen_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::ScreenButton)));
      telos_screen_state=false;
    }
  }
}


void MainWidget::muteStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::MuteButton)) {
      telos_mute_button->
	setPalette(ml_config->palette(MlConfig::MuteButtonColor));
      telos_mute_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::MuteButton)));
      telos_mute_state=true;
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::MuteButton)) {
      telos_mute_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
      telos_mute_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::MuteButton)));
      telos_mute_state=false;
    }
  }
}


void MainWidget::conferenceStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::ConfButton)) {
      telos_conf_button->
	setPalette(ml_config->palette(MlConfig::ConfButtonColor));
      telos_conf_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::ConfButton)));
      telos_conf_state=true;
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::ConfButton)) {
      telos_conf_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
      telos_conf_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::ConfButton)));
      telos_conf_state=false;
    }
  }
}


void MainWidget::vipStateData(unsigned id,bool state)
{
  if(state) {
    if(ml_config->buttonShow(MlConfig::VipButton)) {
      telos_vip_button->
	setPalette(ml_config->palette(MlConfig::VipButtonColor));
      telos_vip_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::VipButton)));
      telos_vip_state=true;
    }
  }
  else {
    if(ml_config->buttonShow(MlConfig::VipButton)) {
      telos_vip_button->
	setPalette(ml_config->palette(MlConfig::IdleColor));
      telos_vip_button->
	setAccel(QKeySequence(ml_config->
			      buttonKey(MlConfig::VipButton)));
      telos_vip_state=false;
    }
  }
}


void MainWidget::receivedStringData(unsigned id,BusDriver::Element elem,
				    unsigned line,const QString &str)
{
  switch(elem) {
      case BusDriver::LookupGlobalElement:
	telos_message_edit->clickedData(0,0);
	break;

      case BusDriver::MessageGlobalElement:
	if(telos_message_data->update(elem,str)) {
	  if(ml_config->msgboxShow()) {
	    telos_message_edit->setMessage(telos_message_data->nick()+
					   QString("> ")+
					   telos_message_data->message());
	  }
	}
	break;

      case BusDriver::LineWarningGlobalElement:
	if(ml_config->lineShowCallerBox(line)) {
	  telos_warning_box[line]->show(str);
	  telos_warning_box[line]->
	    setGeometry(ml_config->lineCallerBoxWindow(line).x()+
			frameGeometry().x()+2,
			ml_config->lineCallerBoxWindow(line).y()+
			frameGeometry().y(),
			ml_config->callerSize().width(),
			ml_config->callerSize().height());
	}
	break;

      default:
	if(telos_caller_data[line]->update(elem,str)) {
	  UpdateCallerBox(line);
	  update(ml_config->lineCallerBoxWindow(line));
	}
	break;
  }
}


void MainWidget::metadataUpdateRequestedData(unsigned id,
					     ClientData::ClientType type)
{
  if(type!=ClientData::TypeUser) {
    return;
  }
  for(unsigned i=0;i<MAX_LINES;i++) {
    telos_caller_data[i]->sendData(telos_driver,i);
  }
}


void MainWidget::requestMetadataData()
{
  ClientData *data=NULL;

  //
  // Find the longest-running peer
  //
  for(unsigned i=1;i<telos_driver->clientList()->size();i++) {
    if(telos_driver->clientList()->at(i)->clientType()==ClientData::TypeUser) {
      if(data==NULL) {
	data=telos_driver->clientList()->at(i);
      }
      else {
	if(telos_driver->clientList()->at(i)->loginDateTime()<
	   data->loginDateTime()) {
	  data=telos_driver->clientList()->at(i);
	}
      }
    }
  }

  //
  // Request an Update
  //
  if(data!=NULL) {
    telos_driver->requestMetadataUpdate(data);
  }
}


void MainWidget::TickClock()
{
  static int ticks=0;
  static int scale=0;
  bool tick_timers=false;
  

  if(ticks++==9) {
    ticks=0;
    tick_timers=true;
  }

  if(scale++==1) {
    scale=0;
    for(int i=0;i<10;i++) {
      if(telos_icon_modulo[i]>1) {
	if(++telos_icon_frame[i]==telos_icon_modulo[i]) {
	  telos_icon_frame[i]=0;
	}
      }
    }
    for(int j=0;j<MAX_LINES;j++) {
      switch(telos_status[j]) {
      case BusDriver::LineIdle:
	if(telos_icon_modulo[0]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][0][telos_icon_frame[0]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][0][telos_icon_frame[0]]);
	  }
	}
	break;
	
      case BusDriver::LineInactive:
	break;
	
      case BusDriver::LineElsewhere:
	if(telos_icon_modulo[4]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][4][telos_icon_frame[4]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][4][telos_icon_frame[4]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineOnAir:
	if(telos_icon_modulo[1]>1) {
	  if(telos_line[0][j]!=NULL) {
	    if(telos_bank[j]==0) {
	      telos_line[0][j]->
		setPixmap(telos_cap[j][1][telos_icon_frame[1]]);
	    }
	    else {
	      telos_line[0][j]->
		setPixmap(telos_cap[j][4][telos_icon_frame[1]]);
	    }
	  }
	  if(telos_line[1][j]!=NULL) {
	    if(telos_bank[j]==0) {
	      telos_line[1][j]->
		setPixmap(telos_cap[j][4][telos_icon_frame[4]]);
	    }
	    else {
	      telos_line[1][j]->
		setPixmap(telos_cap[j][2][telos_icon_frame[4]]);
	    }
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::AirTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::AirTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::AirTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineLocked:
	if((telos_icon_modulo[3]>1)||(telos_icon_modulo[4]>1)) {
	  if(telos_line[0][j]!=NULL) {
	    if(telos_bank[j]==0) {
	      telos_line[0][j]->
		setPixmap(telos_cap[j][3][telos_icon_frame[3]]);
	    }
	    else {
	      telos_line[0][j]->
		setPixmap(telos_cap[j][4][telos_icon_frame[4]]);
	    }
	  }
	  if(telos_line[1][j]!=NULL) {
	    if(telos_bank[j]==1) {
	      telos_line[1][j]->
		setPixmap(telos_cap[j][3][telos_icon_frame[3]]);
	    }
	    else {
	      telos_line[1][j]->
		setPixmap(telos_cap[j][4][telos_icon_frame[4]]);
	    }
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::AirTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::AirTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::AirTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineSpeaker:
	if(telos_icon_modulo[4]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][12][telos_icon_frame[12]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][12][telos_icon_frame[12]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineHandset:
	if(telos_icon_modulo[4]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][11][telos_icon_frame[11]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][11][telos_icon_frame[11]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineHold:
	if(telos_icon_modulo[5]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][5][telos_icon_frame[5]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][5][telos_icon_frame[5]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineScreenedHold:
	if(telos_icon_modulo[6]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][6][telos_icon_frame[6]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][6][telos_icon_frame[6]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineNext:
	if(telos_icon_modulo[7]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][7][telos_icon_frame[7]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][7][telos_icon_frame[7]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::HoldTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::HoldTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::HoldTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineRinging:
	if(telos_icon_modulo[8]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][8][telos_icon_frame[8]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][8][telos_icon_frame[8]]);
	  }
	}
	if(tick_timers) {
	  telos_caller_data[j]->
	    setElement(BusDriver::RingTimerElement,
		       telos_caller_data[j]->
		       element(BusDriver::RingTimerElement).toInt()+1);
	  UpdateTimer(j,telos_caller_data[j]->
		      element(BusDriver::RingTimerElement).toInt());	  
	}
	break;
	
      case BusDriver::LineBusied:
	if(telos_icon_modulo[9]>1) {
	  if(telos_line[0][j]!=NULL) {
	    telos_line[0][j]->
	      setPixmap(telos_cap[j][9][telos_icon_frame[9]]);
	  }
	  if(telos_line[1][j]!=NULL) {
	    telos_line[1][j]->
	      setPixmap(telos_cap[j][9][telos_icon_frame[9]]);
	  }
	}
	break;
      }
      if(ml_config->lineShow(0,j)) {
	telos_line[0][j]->setAccel(QKeySequence(ml_config->lineKey(0,j)));
      }
      if(ml_config->lineShow(1,j)) {
	telos_line[1][j]->setAccel(QKeySequence(ml_config->lineKey(1,j)));
      }
    }
  }
  if(ml_config->buttonShow(MlConfig::RecordButton)&&
     telos_record_running&&ml_config->animateIcons()) {
    if(++telos_record_frame>=telos_record_modulo) {
      telos_record_frame=0;
    }
    telos_record_button->setPixmap(telos_record_icon[telos_record_frame]);
    telos_record_button->
      setAccel(ml_config->buttonKey(MlConfig::RecordButton));
  }
}


void MainWidget::busyData()
{
  telos_driver->busyAll();
}


void MainWidget::dumpPressedData()
{
  telos_driver->dumpDelay(true);
}


void MainWidget::dumpReleasedData()
{
  telos_driver->dumpDelay(false);
}


void MainWidget::recordData()
{
  telos_driver->toggleRecord();
}


void MainWidget::nextData()
{
  telos_driver->takeNext(0);
}


void MainWidget::keypadData()
{
  Keybox *keybox=new Keybox(telos_driver,this,"keybox");
  keybox->exec();
  delete keybox;
}


void MainWidget::flashData()
{
  telos_driver->flash();
}


void MainWidget::screenData()
{
  telos_driver->screen(!telos_screen_state);
}


void MainWidget::muteData()
{
  telos_driver->mute(!telos_mute_state);
}


void MainWidget::confData()
{
  telos_driver->conference(!telos_conf_state);
}


void MainWidget::vipData()
{
  telos_driver->vip(!telos_vip_state);
}


void MainWidget::clientData()
{
  telos_list_clients->show();
}


void MainWidget::UpdateTimer(int line,int value)
{
  int min,sec;

  if(ml_config->lineShowCallerBox(line)) {
    min=value/60;
    sec=value-min*60;
    QPainter *p=new QPainter(this);
    p->setPen(QColor(black));
    p->setFont(ml_config->callerFont(MlConfig::CallerTimer));
    p->fillRect(ml_config->lineCallerBoxWindow(line).x()+
		ml_config->callerWindow(MlConfig::CallerTimer).x(),
		ml_config->lineCallerBoxWindow(line).y()+
		ml_config->callerWindow(MlConfig::CallerTimer).y(),55,
		-ml_config->callerFont(MlConfig::CallerTimer).pixelSize(),
		QColor(ml_config->color(MlConfig::CallerBoxColor)));
    p->drawText(ml_config->lineCallerBoxWindow(line).x()+
		ml_config->callerWindow(MlConfig::CallerTimer).x(),
		ml_config->lineCallerBoxWindow(line).y()+
		ml_config->callerWindow(MlConfig::CallerTimer).y(),
	QString().sprintf("%02d:%02d",min,sec));
    p->end();
    delete p;
  }
}


void MainWidget::SendCallData(int line,bool timers)
{
  telos_caller_data[line]->sendData(telos_driver,line);
}


bool MainWidget::InitLogging()
{
  if(ml_config->logEnable()) {
    switch(ml_config->logType()) {
	case MlConfig::Disk:
	  return true;
	  break;
	  
	case MlConfig::Sql:
	  return InitSql();
	  break;
    }
  }
  return true;
}

void MainWidget::LogCall(int line)
{
  static bool started[MAX_LINES]={false,false,false,false,false,false,
				  false,false,false,false,false,false};

  if(ml_config->logEnable()) {
    if(started[line]) {
      switch(ml_config->logType()) {
	  case MlConfig::Disk:
	    LogDisk(line);
	    break;

	  case MlConfig::Sql:
	    LogSql(line);
	    break;
      }
    }
    else {
      started[line]=true;
    }
  }
}


void MainWidget::ClearCallerBox(int line)
{
  if(ml_config->lineShowCallerBox(line)) {
    if(ml_config->callerShow(MlConfig::CallerName)) {
      telos_caller_name_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerAge)) {
      telos_caller_age_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerCityState)) {
      telos_caller_citystate_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerZip)) {
      telos_caller_zip_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerStation)) {
      telos_caller_station_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerCallCounter)) {
      telos_caller_callcounter_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerGender)) {
      telos_caller_gender_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerPhone)) {
      telos_caller_phone_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerComment)) {
      telos_caller_comment_label[line]->setText("");
    }
    if(ml_config->callerShow(MlConfig::CallerCell)) {
      telos_caller_cell_label[line]->setPixmap(telos_caller_cell_icon[0]);
    }
    if(ml_config->callerShow(MlConfig::CallerQuality)) {
      telos_caller_quality_label[line]->
	setPixmap(telos_caller_quality_icon[0]);
    }
  }
}


void MainWidget::UpdateCallerBox(int line)
{
  if(ml_config->lineShowCallerBox(line)) {
    if(ml_config->callerShow(MlConfig::CallerName)) {
      telos_caller_name_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::NameElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerAge)) {
      telos_caller_age_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::AgeElement).toString());    
    }
    if(ml_config->callerShow(MlConfig::CallerCityState)) {
      telos_caller_citystate_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::CityElement).toString()+", "+
		telos_caller_data[line]->
		element(BusDriver::StateElement).toString());    
      if(telos_caller_data[line]->
	 element(BusDriver::StateElement).toString().isEmpty()||
	 (telos_caller_data[line]->
	  element(BusDriver::StateElement).toString()==" ")) {
	telos_caller_citystate_label[line]->
	  setText(telos_caller_data[line]->
		  element(BusDriver::CityElement).toString());
      }
      if(telos_caller_data[line]->
	 element(BusDriver::CityElement).toString().isEmpty()||
	 (telos_caller_data[line]->
	  element(BusDriver::CityElement).toString()==" ")) {
	telos_caller_citystate_label[line]->
	  setText(telos_caller_data[line]->
		  element(BusDriver::CityElement).toString());
      }
    }
    if(ml_config->callerShow(MlConfig::CallerZip)) {
      telos_caller_zip_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::ZipElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerGender)) {
      telos_caller_gender_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::GenderElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerStation)) {
      telos_caller_station_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::StationElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerPhone)) {
      telos_caller_phone_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::PhoneElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerComment)) {
      telos_caller_comment_label[line]->
	setText(telos_caller_data[line]->
		element(BusDriver::CommentElement).toString());
    }
    if(ml_config->callerShow(MlConfig::CallerCallCounter)) {
      if(telos_caller_data[line]->
	 element(BusDriver::CounterElement).toInt()>0) {
	telos_caller_callcounter_label[line]->
	  setText(QString().sprintf("#%d",telos_caller_data[line]->
				    element(BusDriver::CounterElement).
				    toInt()));
      }
      else {
	telos_caller_callcounter_label[line]->setText("");
      }	
    }
    if(ml_config->callerShow(MlConfig::CallerCell)) {
      if(telos_caller_data[line]->
	 element(BusDriver::CellphoneElement).toString()=="Y") {
	telos_caller_cell_label[line]->setPixmap(telos_caller_cell_icon[1]);
      }
      else {
	telos_caller_cell_label[line]->
	  setPixmap(telos_caller_cell_icon[0]);
      }
    }
    if(ml_config->callerShow(MlConfig::CallerQuality)) {
      switch(telos_caller_data[line]->
	     element(BusDriver::QualityElement).toInt()) {
	  case 1:  // Very Poor
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[1]);
	    break;

	  case 2:  // Poor
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[2]);
	    break;

	  case 3:  // Fair
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[3]);
	    break;

	  case 4:  // Good
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[4]);
	    break;

	  case 5:  // Excellent
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[5]);
	    break;

	  default:  // No icon
	    telos_caller_quality_label[line]->
	      setPixmap(telos_caller_quality_icon[0]);
	    break;
      }
    }
  }
}


void MainWidget::SetCaption()
{
  QString ddname=telos_connection->consoleName();
  switch(telos_director_mode) {
      case ConsoleData::Talent:
	setCaption(QString().sprintf("%s - [Mode: Talent, Console: %s]",
				     (const char *)telos_connection->name(),
				     (const char *)ddname));
	break;

      case ConsoleData::Producer:
	setCaption(QString().sprintf("%s - [Mode: Producer, Console: %s]",
				     (const char *)telos_connection->name(),
				     (const char *)ddname));
	break;

      case ConsoleData::Unavailable:
	setCaption(QString().sprintf("%s - [Mode: Invalid, Console: %s]",
				     (const char *)telos_connection->name(),
				     (const char *)ddname));
  }
}


int main(int argc,char *argv[])
{
  Qt::WFlags f=0;

  QApplication a(argc,argv);
#ifndef WIN32
  for(int i=0;i<argc;i++) {
    if((!strcmp(argv[i],"-v"))||(!strcmp(argv[i],"--version"))) {
      printf("CallCommander User Interface - version %s\n",VERSION);
      exit(0);
    }
  }
#endif  // WIN32

  RProfile *parser=new RProfile();
  parser->setSource(GetConfFile(argc,argv));
  if(parser->boolValue("Global","KeepOnTop",DEFAULT_ON_TOP)) {
    f=Qt::WStyle_StaysOnTop;
  }
  delete parser;

  MainWidget *w=new MainWidget(NULL,"main",f);
  a.setMainWidget(w);
  w->show();
  return a.exec();
}
