// rdconfig.cpp
//
// A container class for a Rivendell Base Configuration
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdconfig.cpp,v 1.13 2007/02/14 21:48:41 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#ifndef WIN32
#include <sys/types.h>
#include <pwd.h>
#include <grp.h>
#endif

#include <qmessagebox.h>
#include <qsettings.h>

#include <rprofile.h>

#include <rdconfig.h>


RDConfig::RDConfig()
{
  clear();
}


RDConfig::RDConfig(QString filename)
{
  clear();
  conf_filename=filename;
}


QString RDConfig::filename() const
{
  return conf_filename;
}


void RDConfig::setFilename(QString filename)
{
  conf_filename=filename;
}


QString RDConfig::mysqlHostname() const
{
  return conf_mysql_hostname;
}


QString RDConfig::mysqlUsername() const
{
  return conf_mysql_username;
}


QString RDConfig::mysqlDbname() const
{
  return conf_mysql_dbname;
}


QString RDConfig::mysqlPassword() const
{
  return conf_mysql_password;
}


QString RDConfig::mysqlDriver() const
{
  return conf_mysql_driver;
}


int RDConfig::mysqlHeartbeatInterval() const
{
  return conf_mysql_heartbeat_interval;
}


QString RDConfig::stationName() const
{
  return conf_station_name;
}


QString RDConfig::password() const
{
  return conf_password;
}


QString RDConfig::audioOwner() const
{
  return conf_audio_owner;
}


QString RDConfig::audioGroup() const
{
  return conf_audio_group;
}


QString RDConfig::ripcdLogname() const
{
  return conf_ripcd_logname;
}


QString RDConfig::airplayLogname() const
{
  return conf_airplay_logname;
}


QString RDConfig::catchdLogname() const
{
  return conf_catchd_logname;
}


int RDConfig::jackConnections() const
{
  return conf_jack_ports[0].size();
}


QString RDConfig::jackPort(int num,int endpt) const
{
  return conf_jack_ports[num][endpt];
}


bool RDConfig::useStreamMeters() const
{
  return conf_use_stream_meters;
}


unsigned RDConfig::sampleRate() const
{
  return conf_sample_rate;
}


unsigned RDConfig::channels() const
{
  return conf_channels;
}


#ifndef WIN32
uid_t RDConfig::uid() const
{
  return conf_uid;
}


gid_t RDConfig::gid() const
{
  return conf_gid;
}
#endif


QString RDConfig::caeLogfile() const
{
  return conf_cae_logfile;
}


bool RDConfig::enableMixerLogging() const
{
  return conf_enable_mixer_logging;
}


bool RDConfig::useRealtime()
{
  return conf_use_realtime;
}


int RDConfig::realtimePriority()
{
  return conf_realtime_priority;
}


QString RDConfig::sasStation() const
{
  return conf_sas_station;
}


int RDConfig::sasMatrix() const
{
  return conf_sas_matrix;
}


unsigned RDConfig::sasBaseCart() const
{
  return conf_sas_base_cart;
}


QString RDConfig::sasTtyDevice() const
{
  return conf_sas_tty_device;
}


QString RDConfig::destination(unsigned n)
{
  if(n>=conf_destinations.size()) {
    return QString();
  }
  return conf_destinations[n];
}


void RDConfig::load()
{
  char sname[256];
#ifndef WIN32
  struct passwd *user;
  struct group *groups;
#endif 

  RProfile *profile=new RProfile();
  profile->setSource(conf_filename);
#ifdef WIN32
  strcpy(sname,"windows");
#else
  gethostname(sname,255);
#endif
  conf_station_name=
    profile->stringValue("Identity","StationName",sname);
  conf_password=profile->stringValue("Identity","Password","");
  conf_audio_owner=
    profile->stringValue("Identity","AudioOwner",RD_DEFAULT_AUDIO_OWNER);
  conf_audio_group=
    profile->stringValue("Identity","AudioGroup",RD_DEFAULT_AUDIO_GROUP);
  conf_mysql_hostname=
    profile->stringValue("mySQL","Hostname",DEFAULT_MYSQL_HOSTNAME);
  conf_mysql_username=
    profile->stringValue("mySQL","Loginname",DEFAULT_MYSQL_USERNAME);
  conf_mysql_dbname=
    profile->stringValue("mySQL","Database",DEFAULT_MYSQL_DATABASE);
  conf_mysql_password=profile->stringValue("mySQL","Password",conf_password);
  conf_mysql_driver=
    profile->stringValue("mySQL","Driver",DEFAULT_MYSQL_DRIVER);
  conf_mysql_heartbeat_interval=
    profile->intValue("mySQL","HeartbeatInterval",
		      DEFAULT_MYSQL_HEARTBEAT_INTERVAL);
  conf_ripcd_logname=profile->stringValue("Ripcd","Logfile","");
  conf_airplay_logname=profile->stringValue("RDAirPlay","Logfile","");
  conf_catchd_logname=profile->stringValue("RDCatchd","Logfile","");
  conf_use_stream_meters=profile->boolValue("Hacks","UseStreamMeters",false);
  conf_sample_rate=profile->intValue("Format","SampleRate",
				     RD_DEFAULT_SAMPLE_RATE);
  conf_channels=profile->intValue("Format","Channels",RD_DEFAULT_CHANNELS);
#ifndef WIN32
  if((user=getpwnam(profile->stringValue("Identity","AudioOwner")))!=NULL) {
    conf_uid=user->pw_uid;
  }
  if((groups=getgrnam(profile->stringValue("Identity","AudioGroup")))!=NULL) {
    conf_gid=groups->gr_gid;
  }
#endif
  conf_cae_logfile=profile->stringValue("Caed","Logfile","");
  conf_enable_mixer_logging=profile->boolValue("Caed","EnableMixerLogging");
  conf_use_realtime=profile->boolValue("Tuning","UseRealtime",false);
  conf_realtime_priority=profile->intValue("Tuning","RealtimePriority",9);
  conf_sas_station=profile->stringValue("SASFilter","Station","");
  conf_sas_matrix=profile->intValue("SASFilter","Matrix",0);
  conf_sas_base_cart=profile->intValue("SASFilter","BaseCart",0);
  conf_sas_tty_device=profile->stringValue("SASFilter","TtyDevice","");
  int c=1;
  QString dest;
  while(!(dest=profile->
	  stringValue("RDBackup",QString().
		      sprintf("Destination%d",c++),"")).isEmpty()) {
    conf_destinations.push_back(dest);
  }
  delete profile;
}


void RDConfig::clear()
{
#ifdef WIN32
  QSettings settings;
  settings.insertSearchPath(QSettings::Windows,"/SalemRadioLabs");
  conf_filename=QString().sprintf("%s\\%s",
				  (const char *)settings.
				  readEntry("/Rivendell/InstallDir"),
				  (const char *)RD_WIN_CONF_FILE);  
#else
  conf_filename=RD_CONF_FILE;
#endif
  conf_mysql_hostname="";
  conf_mysql_username="";
  conf_mysql_dbname="";
  conf_mysql_password="";
  conf_mysql_driver="";
  conf_mysql_heartbeat_interval=DEFAULT_MYSQL_HEARTBEAT_INTERVAL;
  conf_station_name="";
  conf_password="";
  conf_audio_owner="";
  conf_audio_group="";
  conf_ripcd_logname="";
  conf_airplay_logname="";
  conf_catchd_logname="";
  conf_jack_ports[0].clear();
  conf_jack_ports[1].clear();
  conf_use_stream_meters=false;
  conf_sample_rate=RD_DEFAULT_SAMPLE_RATE;
  conf_channels=RD_DEFAULT_CHANNELS;
#ifndef WIN32
  conf_uid=65535;
  conf_gid=65535;
#endif
  conf_cae_logfile="";
  conf_enable_mixer_logging=false;
  conf_use_realtime=false;
  conf_realtime_priority=9;
  conf_sas_station="";
  conf_sas_matrix=-1;
  conf_sas_base_cart=1;
  conf_sas_tty_device="";
  conf_destinations.clear();
}
