// rdsound_panel.cpp
//
// The sound panel widget for RDAirPlay
//
//   (C) Copyright 2002-2004 Fred Gleason <fredg@paravelsystems.com>
//
//      $Id: rdsound_panel.cpp,v 1.35 2007/02/19 17:29:57 fredg Exp $
//
//   This program is free software; you can redistribute it and/or modify
//   it under the terms of the GNU General Public License version 2 as
//   published by the Free Software Foundation.
//
//   This program is distributed in the hope that it will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public
//   License along with this program; if not, write to the Free Software
//   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include <qsignalmapper.h>

#include <rtransportbutton.h>

#include <rdlog_line.h>
#include <rdsound_panel.h>
#include <rdbutton_dialog.h>
#include <rdmacro.h>
#include <rdcut.h>


RDSoundPanel::RDSoundPanel(int station_panels,int user_panels,bool flash,
			   RDEventPlayer *player,RDRipc *ripc,RDCae *cae,
			   RDStation *station,QWidget *parent,const char *name)
  : QWidget(parent,name)
{
  panel_type=RDAirPlayConf::StationPanel;
  panel_number=0;
  panel_setup_mode=false;
  panel_reset_mode=false;
  panel_parent=parent;
  panel_cae=cae;
  panel_user=NULL;
  panel_ripc=ripc;
  panel_station=station;
  panel_station_panels=station_panels;
  panel_user_panels=user_panels;
  panel_event_player=player;
  panel_action_mode=RDAirPlayConf::Normal;
  for(int i=0;i<RD_MAX_STREAMS;i++) {
    panel_active_buttons[i]=NULL;
  }
  panel_flash=flash;
  panel_flash_count=0;
  panel_flash_state=false;
  panel_config_panels=false;
  panel_pause_enabled=false;
  for(unsigned i=0;i<PANEL_MAX_OUTPUTS;i++) {
    panel_card[i]=-1;
    panel_port[i]=-1;
  }

  //
  // Create Fonts
  //
  QFont button_font=QFont("Helvetica",14,QFont::Bold);
  button_font.setPixelSize(14);

  //
  // Load Buttons
  //
  panel_mapper=new QSignalMapper(this,"panel_mapper");
  connect(panel_mapper,SIGNAL(mapped(int)),this,SLOT(buttonMapperData(int)));
  LoadPanels();

  //
  // Back Button
  //
  panel_down_button=new RTransportButton(RTransportButton::Rewind,
					 this,"setup_button");
  panel_down_button->
    setGeometry(0,(15+PANEL_BUTTON_SIZE_Y)*PANEL_BUTTON_COLUMNS,
		PANEL_BUTTON_SIZE_X,50);
  panel_down_button->setDisabled(true);
 panel_down_button->setFocusPolicy(QWidget::NoFocus);
  connect(panel_down_button,SIGNAL(clicked()),this,SLOT(panelDown()));

  //
  // Forward Button
  //
  panel_up_button=new RTransportButton(RTransportButton::FastForward,
				       this,"setup_button");
  panel_up_button->setGeometry(PANEL_BUTTON_SIZE_X+15,
			       (15+PANEL_BUTTON_SIZE_Y)*PANEL_BUTTON_COLUMNS,
			       PANEL_BUTTON_SIZE_X,50);
  connect(panel_up_button,SIGNAL(clicked()),this,SLOT(panelUp()));
  panel_up_button->setDisabled(!((panel_station_panels + panel_user_panels) > 1));  

  //
  // Panel Label
  //
  panel_label=new QLabel(this,"panel_label");
  panel_label->setGeometry(2*(PANEL_BUTTON_SIZE_X+15),
		       (15+PANEL_BUTTON_SIZE_Y)*PANEL_BUTTON_COLUMNS,
		       PANEL_BUTTON_SIZE_X,50);
  panel_label->setFont(button_font);
  panel_label->setAlignment(AlignCenter);
 panel_label->setFocusPolicy(QWidget::NoFocus);
  if(panel_station_panels>0) {
    panel_number=0;
    panel_type=RDAirPlayConf::StationPanel;
    SetPanelLabel();
    panel_buttons[0].show();
  }
  else {
    if(panel_user_panels>0) {
      panel_number=0;
      panel_type=RDAirPlayConf::UserPanel;
      SetPanelLabel();
      panel_buttons[0].show();
    }
    else {
      setDisabled(true);
    }
  }
  
  //
  // Reset Button
  //
  panel_reset_button=new RPushButton(this,"reset_button");
  panel_reset_button->
    setGeometry((15+PANEL_BUTTON_SIZE_X)*(PANEL_BUTTON_ROWS-2),
		(15+PANEL_BUTTON_SIZE_Y)*PANEL_BUTTON_COLUMNS,
		PANEL_BUTTON_SIZE_X,50);
  panel_reset_button->setFont(button_font);
  panel_reset_button->setText(tr("Reset"));
  panel_reset_button->setFlashColor(QColor(RDPANEL_RESET_FLASH_COLOR));
 panel_reset_button->setFocusPolicy(QWidget::NoFocus);
  connect(panel_reset_button,SIGNAL(clicked()),this,SLOT(resetClickedData()));

  //
  // Setup Button
  //
  panel_setup_button=new RPushButton(this,"setup_button");
  panel_setup_button->
    setGeometry((15+PANEL_BUTTON_SIZE_X)*(PANEL_BUTTON_ROWS-1),
		(15+PANEL_BUTTON_SIZE_Y)*PANEL_BUTTON_COLUMNS,
		PANEL_BUTTON_SIZE_X,50);
  panel_setup_button->setFont(button_font);
  panel_setup_button->setText(tr("Setup"));
  panel_setup_button->setFlashColor(QColor(RDPANEL_SETUP_FLASH_COLOR));
 panel_setup_button->setFocusPolicy(QWidget::NoFocus);
  connect(panel_setup_button,SIGNAL(clicked()),this,SLOT(setupClickedData()));

  //
  // Button Dialog Box
  //
  panel_button_dialog=new RDButtonDialog(panel_station->name(),
					 this,"panel_button_dialog");

  //
  // CAE Setup
  //
  connect(panel_cae,SIGNAL(timescalingSupported(int,bool)),
	  this,SLOT(timescalingSupportedData(int,bool)));
}


QSize RDSoundPanel::sizeHint() const
{
  return QSize(500,535);
}


QSizePolicy RDSoundPanel::sizePolicy() const
{
  return QSizePolicy(QSizePolicy::Fixed,QSizePolicy::Fixed);
}


int RDSoundPanel::card(int outnum) const
{
  return panel_card[outnum];
}


void RDSoundPanel::setCard(int outnum,int card)
{
  panel_card[outnum]=card;
  panel_cae->requestTimescale(card);
}


int RDSoundPanel::port(int outnum) const
{
  return panel_port[outnum];
}


void RDSoundPanel::setPort(int outnum,int port)
{
  panel_port[outnum]=port;
}


QString RDSoundPanel::outputText(int outnum) const
{
  return panel_output_text[outnum];
}


void RDSoundPanel::setOutputText(int outnum,const QString &text)
{
  panel_output_text[outnum]=text;
}


void RDSoundPanel::setRmls(int outnum,const QString &start_rml,
			   const QString &stop_rml)
{
  panel_start_rml[outnum]=start_rml;
  panel_stop_rml[outnum]=stop_rml;
}


void RDSoundPanel::setLogName(const QString &logname)
{
  panel_logname=logname;
}


void RDSoundPanel::setSvcName(const QString &svcname)
{
  panel_svcname=svcname;
  panel_svcname.replace(" ","_");
}


void RDSoundPanel::setButton(RDAirPlayConf::PanelType type,int panel,
			   int row,int col,unsigned cartnum)
{
  QString str;

  RDPanelButton *button=
    panel_buttons[PanelOffset(type,panel)].panelButton(row,col);
  if(button->deck()>=0) {
    return;
  }
  button->clear();
  if(cartnum>0) {
    button->setCart(cartnum);
    RDCart *cart=new RDCart(cartnum);
    if(cart->exists()) {
      button->setText(cart->title());
      button->setLength(cart->forcedLength());
    }
    else {
      str=QString(tr("Cart"));
      button->setText(QString().sprintf("%s %06u",(const char *)str,cartnum));
      button->setLength(0);
    }
    delete cart;
  }
  SaveButton(type,panel,row,col);
}


void RDSoundPanel::setLogfile(QString filename)
{
  panel_logfile=filename;
}


void RDSoundPanel::play(RDAirPlayConf::PanelType type,int panel,
			int row, int col,RDLogLine::StartSource src)
{
  PlayButton(type,panel,row,col,src);
}


bool RDSoundPanel::pause(RDAirPlayConf::PanelType type,int panel,
			 int row,int col)
{
  if(panel_pause_enabled) {
    PauseButton(type,panel,row,col);
    return true;
  }
  return false;
}


void RDSoundPanel::stop(RDAirPlayConf::PanelType type,int panel,
			int row,int col)
{
  StopButton(type,panel,row,col);
}


RDAirPlayConf::ActionMode RDSoundPanel::actionMode() const
{
  return panel_action_mode;
}


void RDSoundPanel::setActionMode(RDAirPlayConf::ActionMode mode)
{
  if(panel_setup_mode) {
    return;
  }
  switch(mode) {
      case RDAirPlayConf::CopyFrom:
	mode=RDAirPlayConf::CopyFrom;
	break;

      default:
	mode=RDAirPlayConf::Normal;
	break;
  }
  if(mode!=panel_action_mode) {
    panel_action_mode=mode;
    panel_setup_button->setEnabled(panel_action_mode==RDAirPlayConf::Normal);
    for(unsigned i=0;i<panel_buttons.size();i++) {
      panel_buttons[i].setActionMode(panel_action_mode);
    }
  }
}


bool RDSoundPanel::pauseEnabled() const
{
  return panel_pause_enabled;
}


void RDSoundPanel::setPauseEnabled(bool state)
{
  if(state) {
    panel_reset_button->show();
  }
  else {
    panel_reset_button->hide();
  }
  panel_pause_enabled=state;
}


void RDSoundPanel::changeUser()
{
  if(panel_user!=NULL) {
    delete panel_user;
  }
  panel_user=new RDUser(panel_ripc->user());
  panel_config_panels=panel_user->configPanels();
  LoadPanels();
  panel_buttons[PanelOffset(panel_type,panel_number)].show();
}


void RDSoundPanel::tickClock()
{
  emit tick();
  if(panel_flash) {
    if(panel_flash_count++>1) {
      emit buttonFlash(panel_flash_state);
      panel_flash_state=!panel_flash_state;
      panel_flash_count=0;
    }
  }
}


void RDSoundPanel::panelUp()
{
  panel_buttons[PanelOffset(panel_type,panel_number)].hide();
  panel_down_button->setEnabled(true);
  switch(panel_type) {
      case RDAirPlayConf::StationPanel:
	if(panel_number<(panel_station_panels-1)) {
	  panel_number++;
	  if((panel_number>=(panel_station_panels-1))&&
	     (panel_user_panels==0)){
	    panel_up_button->setDisabled(true);
	  }
	 /* Deal with the one panel case */
	 if (panel_number > panel_station_panels -1)
	  panel_number = panel_station_panels -1;	
	}
	else {
	  if (panel_user_panels>0){
	    panel_type=RDAirPlayConf::UserPanel;
	    panel_number=0;
	    if(panel_number>=(panel_user_panels-1))
	      panel_up_button->setDisabled(true);
	  }
	}
	break;

      case RDAirPlayConf::UserPanel:
	panel_number++;
	if(panel_number>=(panel_user_panels-1)) {
	  panel_up_button->setDisabled(true);
	}
	/* Deal with the one panel case */
	if (panel_number > panel_user_panels -1)
		panel_number = panel_user_panels -1;	
	break;
  }
  SetPanelLabel();
  panel_buttons[PanelOffset(panel_type,panel_number)].show();
}


void RDSoundPanel::panelDown()
{
  panel_buttons[PanelOffset(panel_type,panel_number)].hide();
  panel_up_button->setEnabled(true);
  switch(panel_type) {
      case RDAirPlayConf::StationPanel:
	panel_number--;
        /* Deal with the one panel case */
        if (panel_number < 0)
          panel_number = 0;	
	if(panel_number==0) {
	  panel_down_button->setDisabled(true);
	}
	break;

      case RDAirPlayConf::UserPanel:
	if(panel_number==0) {
	  panel_type=RDAirPlayConf::StationPanel;
	  panel_number=panel_station_panels-1;
	  if (panel_number == 0)
	    panel_down_button->setDisabled(true);
	}
	else {
	  panel_number--;
	  if((panel_number<=0)&&(panel_station_panels==0)) {
	    panel_down_button->setDisabled(true);
	  }
	}
        /* Deal with the one panel case */
        if (panel_number < 0)
          panel_number = 0;	
	break;
  }
  SetPanelLabel();
  panel_buttons[PanelOffset(panel_type,panel_number)].show();
}


void RDSoundPanel::resetClickedData()
{
  if(panel_reset_mode) {
    panel_reset_mode=false;
    panel_reset_button->setFlashingEnabled(false);
    panel_setup_button->setEnabled(true);
  }
  else {
    panel_reset_mode=true;
    panel_reset_button->setFlashingEnabled(true);
    panel_setup_button->setDisabled(true);
  }
}


void RDSoundPanel::setupClickedData()
{
  if(panel_setup_mode) {
    panel_setup_mode=false;
    panel_setup_button->setFlashingEnabled(false);
    panel_reset_button->setEnabled(true);
  }
  else {
    panel_setup_mode=true;
    panel_setup_button->setFlashingEnabled(true);
    panel_reset_button->setDisabled(true);
  }
}


void RDSoundPanel::buttonMapperData(int id)
{
  int row=id/PANEL_BUTTON_COLUMNS;
  int col=id-row*PANEL_BUTTON_COLUMNS;
  unsigned cartnum;

  switch(panel_action_mode) {
      case RDAirPlayConf::CopyFrom:
	if((cartnum=panel_buttons[PanelOffset(panel_type,panel_number)].
	    panelButton(row,col)->cart())>0) {
	  emit selectClicked(cartnum);
	}
	break;
	
      default:
	if(panel_setup_mode) {
	  if((panel_type==RDAirPlayConf::StationPanel)&&
	     (!panel_config_panels)) {
	    ClearReset();
	    return;
	  }
	  if(panel_button_dialog->
	     exec(panel_buttons[PanelOffset(panel_type,panel_number)].
		  panelButton(row,col))==0) {
	    SaveButton(panel_type,panel_number,row,col);
	  }
	}
	else {
	  RDPlayDeck *deck=panel_buttons[PanelOffset(panel_type,panel_number)].
	    panelButton(row,col)->playDeck();
	  if(panel_reset_mode) {
	    if(deck!=NULL) {
	      switch(deck->state()) {
		  case RDPlayDeck::Playing:
		  case RDPlayDeck::Paused:
		    StopButton(panel_type,panel_number,row,col);
		    break;
		    
		  default:
		    break;
	      }
	    }
	  }
	  else {
	    if(deck==NULL) {
	      PlayButton(panel_type,panel_number,row,col,
			 RDLogLine::StartManual);
	    }
	    else {
	      if(panel_pause_enabled) {
		if(deck->state()!=RDPlayDeck::Paused) {
		  PauseButton(panel_type,panel_number,row,col);
		}
		else {
		  PlayButton(panel_type,panel_number,row,col,
			     RDLogLine::StartManual);
		}
	      }
	      else {
		StopButton(panel_type,panel_number,row,col);
	      }
	    }
	  }
	}
  }
  ClearReset();
}


void RDSoundPanel::stateChangedData(int id,RDPlayDeck::State state)
{
  switch(state) {
      case RDPlayDeck::Playing:
	Playing(id);
	break;

      case RDPlayDeck::Stopped:
      case RDPlayDeck::Finished:
	Stopped(id);
	break;

      case RDPlayDeck::Paused:
	Paused(id);
	break;

      default:
	break;
  }
}


void RDSoundPanel::positionData(int id,int pos)
{
}


void RDSoundPanel::timescalingSupportedData(int card,bool state)
{
  for(unsigned i=0;i<PANEL_MAX_OUTPUTS;i++) {
    if(card==panel_card[i]) {
      panel_timescaling_supported[i]=state;
    }
  }
}


void RDSoundPanel::PlayButton(RDAirPlayConf::PanelType type,int panel,
			    int row,int col,RDLogLine::StartSource src)
{
  RDPlayDeck *deck=
    panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->playDeck();
  if(deck!=NULL) {
    deck->play(deck->currentPosition());
    panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->
      setStartTime(QTime::currentTime().addMSecs(panel_station->timeOffset()).
	addMSecs(-deck->currentPosition()));
    return;
  }

  int cartnum=0;

  if((cartnum=panel_buttons[PanelOffset(type,panel)].
      panelButton(row,col)->cart())==0) {
    LogLine(QString().sprintf("Tried to start empty button.  Row=%d, Col=%d",
			      row,col));
    return;
  }
  RDCart *cart=new RDCart(cartnum);
  if(!cart->exists()) {
    delete cart;
    LogLine(QString().sprintf("Tried to start non-existent cart: %u",cartnum));
    return;
  }
  panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->
    setStartSource(src);
  switch(cart->type()) {
      case RDCart::Audio:
	PlayAudio(panel_buttons[PanelOffset(type,panel)].
		  panelButton(row,col),cart);
	break;

      case RDCart::Macro:
	PlayMacro(panel_buttons[PanelOffset(type,panel)].
		  panelButton(row,col),cart);
	break;

      default:
	break;
  }
  delete cart;
}


bool RDSoundPanel::PlayAudio(RDPanelButton *button,RDCart *cart)
{
  RDLogLine logline;

  bool timescale=false;
  int button_deck=GetFreeButtonDeck();
  if(button_deck<0) {
    LogLine(QString().
	    sprintf("No button deck available, playout aborted.  Cart=%u",
		    cart->number()));
    return false;
  }
  button->setOutput(GetFreeOutput());
  button->setOutputText(panel_output_text[button->output()]);
  button->setPlayDeck(new RDPlayDeck(panel_cae,button_deck,this));
  button->playDeck()->setCard(panel_card[button->output()]);
  button->playDeck()->setPort(panel_port[button->output()]);
  if(panel_timescaling_supported&&cart->enforceLength()) {
    timescale=true;
  }
  logline.loadCart(cart->number(),RDLogLine::Play,0,timescale);
  if(!button->playDeck()->setCart(&logline,true)) {
    delete button->playDeck();
    button->setPlayDeck(NULL);
    LogLine(QString().
	    sprintf("No CAE stream available, playout aborted.  Cart=%u",
		    cart->number()));
    return false;
  }
  button->setCutName(logline.cutName());
  panel_active_buttons[button_deck]=button;

  //
  // Set Mappings
  //
  connect(button->playDeck(),SIGNAL(stateChanged(int,RDPlayDeck::State)),
	  this,SLOT(stateChangedData(int,RDPlayDeck::State)));
  connect(button->playDeck(),SIGNAL(position(int,int)),
	  this,SLOT(positionData(int,int)));
  connect(this,SIGNAL(tick()),button,SLOT(tickClock()));
  
  //
  // Start Playout
  //
  button->
    setStartTime(QTime::currentTime().addMSecs(panel_station->timeOffset()));
  if(timescale) {
    button->setLength(cart->forcedLength());
  }
  else {
    button->setLength(button->playDeck()->cut()->length());
  }
  button->playDeck()->play(0);
  panel_event_player->
    exec(logline.resolveWildcards(panel_start_rml[button->output()]));
  return true;
}


void RDSoundPanel::PlayMacro(RDPanelButton *button,RDCart *cart)
{
  RDMacro rml;
  rml.setRole(RDMacro::Cmd);
  rml.setAddress(panel_station->address());
  rml.setEchoRequested(false);
  rml.setCommand(RDMacro::EX);
  rml.setArgQuantity(1);
  rml.setArg(0,cart->number());
  panel_ripc->sendRml(&rml);
  if(!panel_logname.isEmpty()) {
    LogTrafficMacro(button);
  }
}


void RDSoundPanel::PauseButton(RDAirPlayConf::PanelType type,int panel,
			       int row,int col)
{
  if (panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->
      playDeck()){
    panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->
      playDeck()->pause();

    panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->
      setStartTime(QTime());
  }
}


void RDSoundPanel::StopButton(RDAirPlayConf::PanelType type,int panel,
			    int row,int col)
{
  RDPlayDeck *deck=
    panel_buttons[PanelOffset(type,panel)].panelButton(row,col)->playDeck();
  if(deck!=NULL) {
    switch(deck->state()) {
	case RDPlayDeck::Playing:
	  deck->stop();
	  break;

	case RDPlayDeck::Paused:
	  deck->clear();
	  break;

	default:
	  break;
    }
  }
}


void RDSoundPanel::LoadPanels()
{
  panel_buttons.clear();

  //
  // Load Buttons
  //
  for(int i=0;i<panel_station_panels;i++) {
    panel_buttons.push_back(RDButtonPanel(panel_station,panel_flash,this));
    for(int j=0;j<PANEL_BUTTON_ROWS;j++) {
      for(int k=0;k<PANEL_BUTTON_COLUMNS;k++) {
	connect(panel_buttons.back().panelButton(j,k),SIGNAL(clicked()),
		panel_mapper,SLOT(map()));
	panel_mapper->setMapping(panel_buttons.back().panelButton(j,k),
			   j*PANEL_BUTTON_COLUMNS+k);
      }
    }
    LoadPanel(RDAirPlayConf::StationPanel,i);
  }
  for(int i=0;i<panel_user_panels;i++) {
    panel_buttons.push_back(RDButtonPanel(panel_station,panel_flash,this));
    for(int j=0;j<PANEL_BUTTON_ROWS;j++) {
      for(int k=0;k<PANEL_BUTTON_COLUMNS;k++) {
	connect(panel_buttons.back().panelButton(j,k),SIGNAL(clicked()),
		panel_mapper,SLOT(map()));
	panel_mapper->setMapping(panel_buttons.back().panelButton(j,k),
			   j*PANEL_BUTTON_COLUMNS+k);
      }
    }
    LoadPanel(RDAirPlayConf::UserPanel,i);
  }
}


void RDSoundPanel::LoadPanel(RDAirPlayConf::PanelType type,int panel)
{
  QString owner;
  int offset=0;

  switch(type) {
      case RDAirPlayConf::UserPanel:
	if(panel_user==NULL) {
	  return;
	}
	owner=panel_user->name();
	offset=panel_station_panels+panel;
	break;

      case RDAirPlayConf::StationPanel:
	owner=panel_station->name();
	offset=panel;
	break;
  }

  QString sql=QString().sprintf("select PANELS.ROW_NO,PANELS.COLUMN_NO,\
    PANELS.LABEL,PANELS.CART,PANELS.DEFAULT_COLOR,CART.FORCED_LENGTH\
    from PANELS left join CART on PANELS.CART=CART.NUMBER\
    where PANELS.TYPE=%d && PANELS.OWNER=\"%s\" && PANELS.PANEL_NO=%d\
    order by PANELS.COLUMN_NO,PANELS.ROW_NO",
				type,
				(const char *)owner,
				panel);
  QSqlQuery *q=new QSqlQuery(sql);
  while(q->next()) {
    panel_buttons[offset].
      panelButton(q->value(0).toInt(),q->value(1).toInt())->
      setText(q->value(2).toString());
    panel_buttons[offset].
      panelButton(q->value(0).toInt(),q->value(1).toInt())->
      setCart(q->value(3).toInt());
    panel_buttons[offset].
      panelButton(q->value(0).toInt(),q->value(1).toInt())->
      setLength(q->value(5).toInt());
    if(q->value(4).toString().isEmpty()) {
      panel_buttons[offset].
	panelButton(q->value(0).toInt(),q->value(1).toInt())->
	setColor(palette().active().background());
      panel_buttons[offset].
	panelButton(q->value(0).toInt(),q->value(1).toInt())->
	setDefaultColor(palette().active().background());
    }
    else {
      panel_buttons[offset].
	panelButton(q->value(0).toInt(),q->value(1).toInt())->
	setColor(QColor(q->value(4).toString()));
      panel_buttons[offset].
	panelButton(q->value(0).toInt(),q->value(1).toInt())->
	setDefaultColor(QColor(q->value(4).toString()));
    }
  }
  delete q;
}


void RDSoundPanel::SaveButton(RDAirPlayConf::PanelType type,
			    int panel,int row,int col)
{
  QString sql;
  QSqlQuery *q;
  QString owner;
  int offset=0;

  switch(type) {
      case RDAirPlayConf::UserPanel:
	owner=panel_user->name();
	offset=panel_station_panels+panel;
	break;

      case RDAirPlayConf::StationPanel:
	owner=panel_station->name();
	offset=panel;
	break;
  }

  //
  // Determine if the button exists
  //
  sql=QString().sprintf("select LABEL from PANELS where \
          TYPE=%d && OWNER=\"%s\" && PANEL_NO=%d && ROW_NO=%d && COLUMN_NO=%d",
			type,
			(const char *)owner,
			panel,
			row,
			col);
  q=new QSqlQuery(sql);
  if(q->size()>0) {
    //
    // If so, update the record
    //
    delete q;
    sql=QString().sprintf("update PANELS set LABEL=\"%s\",\
    CART=%d,DEFAULT_COLOR=\"%s\" where (TYPE=%d)&&(OWNER=\"%s\")&&\
    (PANEL_NO=%d)&&(ROW_NO=%d)&&(COLUMN_NO=%d)",
			  (const char *)panel_buttons[offset].
			  panelButton(row,col)->text(),
			  panel_buttons[PanelOffset(panel_type,panel_number)].
			  panelButton(row,col)->cart(),
			  (const char *)panel_buttons[offset].
			  panelButton(row,col)->defaultColor().
			  name(),
			  type,
			  (const char *)owner,
			  panel,
			  row,
			  col);
    q=new QSqlQuery(sql);
    if(q->isActive()) {
      delete q;
      return;
    }
    delete q;
  }
  else {
    delete q;
    
    //
    // Otherwise, insert a new one
    //
    sql=QString().sprintf("insert into PANELS (TYPE,OWNER,\
    PANEL_NO,ROW_NO,COLUMN_NO,LABEL,CART,DEFAULT_COLOR)\
    values (%d,\"%s\",%d,%d,%d,\"%s\",%d,\"%s\")",
			  type,
			  (const char *)owner,
			  panel,
			  row,
			  col,
			  (const char *)panel_buttons[offset].
			  panelButton(row,col)->text(),
			  panel_buttons[PanelOffset(panel_type,panel_number)].
			  panelButton(row,col)->cart(),
			  (const char *)panel_buttons[offset].
			  panelButton(row,col)->
			  defaultColor().name());
    q=new QSqlQuery(sql);
    delete q;
    
  }
}


void RDSoundPanel::SetPanelLabel()
{
  switch(panel_type) {
      case RDAirPlayConf::StationPanel:
	panel_label->setText(QString().sprintf("Panel S:%d",panel_number+1));
	break;

      case RDAirPlayConf::UserPanel:
	panel_label->setText(QString().sprintf("Panel U:%d",panel_number+1));
	break;
  }
}


int RDSoundPanel::PanelOffset(RDAirPlayConf::PanelType type,int panel)
{
  switch(type) {
      case RDAirPlayConf::StationPanel:
	return panel;
	break;

      case RDAirPlayConf::UserPanel:
	return panel_station_panels+panel;
	break;
  }
  return 0;
}


int RDSoundPanel::GetFreeButtonDeck()
{
  for(int i=0;i<RD_MAX_STREAMS;i++) {
    if(panel_active_buttons[i]==NULL) {
      return i;
    }
  }
  return -1;
}


int RDSoundPanel::GetFreeOutput()
{
  bool active=false;

  for(int i=0;i<PANEL_MAX_OUTPUTS;i++) {
    active=false;
    for(int j=0;j<RD_MAX_STREAMS;j++) {
      if((panel_active_buttons[j]!=NULL)&&
	 (panel_active_buttons[j]->output()==i)) {
	active=true;
      }
    }
    if(!active) {
      return i;
    }
  }
  return PANEL_MAX_OUTPUTS-1;
}


void RDSoundPanel::LogPlayEvent(unsigned cartnum,int cutnum)
{
  RDCut *cut=new RDCut(QString().sprintf("%06u_%03d",cartnum,cutnum));
  cut->logPlayout();
  delete cut;
}


void RDSoundPanel::LogTraffic(RDPanelButton *button)
{
  QString sql;
  QSqlQuery *q;
  QDateTime datetime(QDate::currentDate(),QTime::currentTime());

  sql=QString().sprintf("select CART.TITLE,CART.ARTIST,CART.PUBLISHER,\
                         CART.COMPOSER,CART.USAGE_CODE,CUTS.ISRC,\
                         CART.ALBUM,CART.LABEL \
                         from CART left join CUTS \
                         on CART.NUMBER=CUTS.CART_NUMBER \
                         where CUTS.CUT_NAME=\"%s\"",
			(const char *)button->cutName());
  q=new QSqlQuery(sql);
  if(q->first()) {
    sql=QString().sprintf("insert into `%s_SRT` set\
                         LENGTH=%d,CART_NUMBER=%u,\
                         STATION_NAME=\"%s\",EVENT_DATETIME=\"%s %s\",\
                         EVENT_TYPE=%d,EVENT_SOURCE=%d,PLAY_SOURCE=%d,\
                         CUT_NUMBER=%d,TITLE=\"%s\",ARTIST=\"%s\",\
                         PUBLISHER=\"%s\",COMPOSER=\"%s\",USAGE_CODE=%d,\
                         ISRC=\"%s\",START_SOURCE=%d,ALBUM=\"%s\",\
                         LABEL=\"%s\"",
			  (const char *)panel_svcname,
			  button->startTime().msecsTo(datetime.time()),
			  button->cart(),
			  (const char *)panel_station->name(),
			  (const char *)datetime.toString("yyyy-MM-dd"),
			  (const char *)button->startTime().
			  toString("hh:mm:ss"),
			  RDAirPlayConf::TrafficStop,
			  RDLogLine::SoundPanel,
			  RDLogLine::SoundPanel,
			  button->cutName().right(3).toInt(),
			  (const char *)q->value(0).toString(),
			  (const char *)q->value(1).toString(),
			  (const char *)q->value(2).toString(),
			  (const char *)q->value(3).toString(),
			  q->value(4).toInt(),
			  (const char *)q->value(5).toString(),
			  button->startSource(),
			  (const char *)q->value(6).toString(),
			  (const char *)q->value(7).toString());
    delete q;
    q=new QSqlQuery(sql);
  }
  delete q;
}


void RDSoundPanel::LogTrafficMacro(RDPanelButton *button)
{
  QString sql;
  QSqlQuery *q;
  QDateTime datetime(QDate::currentDate(),QTime::currentTime());

  sql=QString().sprintf("insert into %s_REC set\
                         LENGTH=0,CART_NUMBER=%d,STATION_NAME=\"%s\",\
                         EVENT_DATETIME=\"%s %s\",EVENT_TYPE=%d,\
                         PLAY_SOURCE=%d",
			(const char *)panel_logname,
			button->cart(),
			(const char *)panel_station->name(),
			(const char *)datetime.toString("yyyy-MM-dd hh:mm:ss"),
			(const char *)button->startTime().toString("hh:mm:ss"),
			RDAirPlayConf::TrafficMacro,
			RDLogLine::SoundPanel);
  q=new QSqlQuery(sql);
  delete q;
}


void RDSoundPanel::LogLine(QString str)
{
  FILE *file;

  if(panel_logfile.isEmpty()) {
    return;
  }

  QDateTime current=QDateTime::currentDateTime();
  if((file=fopen(panel_logfile,"a"))==NULL) {
    return;
  }
  fprintf(file,"%02d/%02d/%4d - %02d:%02d:%02d.%03d : RDSoundPanel: %s\n",
	  current.date().month(),
	  current.date().day(),
	  current.date().year(),
	  current.time().hour(),
	  current.time().minute(),
	  current.time().second(),
	  current.time().msec(),
	  (const char *)str);
  fclose(file);
}


void RDSoundPanel::Playing(int id)
{
  if(panel_active_buttons[id]==NULL) {
    LogLine(QString().sprintf("Invalid ID=%d in RDSoundPanel::Playing()",
			      id));
    return;
  }
  panel_active_buttons[id]->setState(true);
  panel_active_buttons[id]->setColor(RDPANEL_PLAY_BACKGROUND_COLOR);
  LogPlayEvent(panel_active_buttons[id]->playDeck()->cart()->number(),
	       panel_active_buttons[id]->playDeck()->cut()->cutNumber());
  LogLine(QString().
	  sprintf("Playout started: id=%d  cart=%u  cut=%d",
		  id,panel_active_buttons[id]->playDeck()->cart()->number(),
		  panel_active_buttons[id]->playDeck()->cut()->cutNumber()));
}


void RDSoundPanel::Paused(int id)
{
  if(panel_active_buttons[id]==NULL) {
    LogLine(QString().sprintf("Invalid ID=%d in RDSoundPanel::Paused()",
			      id));
    return;
  }
  panel_active_buttons[id]->setState(true);
  panel_active_buttons[id]->setColor(RDPANEL_PAUSED_BACKGROUND_COLOR);
  LogLine(QString().
	  sprintf("Playout paused: id=%d  cart=%u  cut=%d",
		  id,panel_active_buttons[id]->playDeck()->cart()->number(),
		  panel_active_buttons[id]->playDeck()->cut()->cutNumber()));
}


void RDSoundPanel::Stopped(int id)
{
  if(panel_active_buttons[id]==NULL) {
    LogLine(QString().sprintf("Invalid ID=%d in RDSoundPanel::Stopped()",
			      id));
    return;
  }
  LogTraffic(panel_active_buttons[id]);
  ClearChannel(id);
  panel_active_buttons[id]->setState(false);
  disconnect(this,SIGNAL(tick()),panel_active_buttons[id],SLOT(tickClock()));
  panel_active_buttons[id]->playDeck()->disconnect();
  delete panel_active_buttons[id]->playDeck();
  panel_active_buttons[id]->setPlayDeck(NULL);
  panel_active_buttons[id]->reset();
  panel_active_buttons[id]=NULL;
  LogLine(QString().sprintf("Playout stopped: id=%d",id));
}


void RDSoundPanel::ClearChannel(int id)
{
  RDPlayDeck *playdeck=panel_active_buttons[id]->playDeck();
  if(panel_cae->
     playPortActive(playdeck->card(),playdeck->port(),playdeck->stream())) {
    return;
  }
  panel_event_player->exec(panel_stop_rml[panel_active_buttons[id]->output()]);
}


void RDSoundPanel::ClearReset()
{
  panel_reset_mode=false;
  panel_reset_button->setFlashingEnabled(false);
  panel_setup_button->setEnabled(true);
}
